// Public fipcore header file
// (C) Copyright Exoligent 2014-2018
// Authors and Contributors: Maxime Coroyer
#ifndef _FIPCORE_H_
#define _FIPCORE_H_

#include <stdint.h>
#include <stdbool.h>

#if defined(_WIN32) || defined(_WIN64)
    #include <windows.h>
    #if defined(FIPCORE_STATIC)
        #define FIPCORE_EXPORT
    #else
        #if defined(FIPCORE_DLL_EXPORT)
            #define FIPCORE_EXPORT __declspec(dllexport)
        #else
            #define FIPCORE_EXPORT __declspec(dllimport)
        #endif
    #endif
#elif defined(__linux__) || defined(__APPLE__) || defined(__QNX__)
    #define FIPCORE_EXPORT 
#endif

//### exoligent enums - concatenation of exocommon.h
#ifndef _EXOCOMMON_H_
#define _EXOCOMMON_H_
typedef enum {
    SUCCESS = 0,
    FIPCODE_BASE_ERROR = 0x60000000,
    FIPCODE_CR_NON_SIGNIFIANT = 0x60000000,
    FIPCODE_CR_NON_PRODUCING,
    FIPCODE_CR_NON_CONSUMING,
    FIPCODE_CR_NON_COMPATIBLE,
    FIPCODE_CR_FIFO_EVT_EMPTY,
    FIPCODE_CR_FIFO_APER_FULL,
    FIPCODE_CR_CMDE_INC,
    FIPCODE_CR_OVERFLOW,
    FIPCODE_CR_FIFO_MSG_EMI_INC,
    FIPCODE_CR_FIFO_MSG_EMI_FULL,
    FIPCODE_CR_FIFO_MSG_EMI_EMPTY,
    FIPCODE_CR_FIFO_MSG_REC_EMPTY,
    FIPCODE_CR_UNDERFLOW,
    FIPCODE_CR_DEF_CMDE_BA,
    FIPCODE_CR_DEF_ACQ_EOC,
    IPROT_SW_WARNING = 0x70000001,
    IPROT_SW_READ_PARTIAL,
    IPROT_SW_FAIL,
    IPROT_SW_EMPTY,
    IPROT_SW_BAD_LENGTH,
    IPROT_SW_NOT_FOUND,
    IPROT_SW_BAD_REQUEST,
    IPROT_SW_BAD_CLA,
    IPROT_SW_BAD_CTL,
    IPROT_SW_BAD_P1,
    IPROT_SW_BAD_P2,
    IPROT_SW_DATA_TOO_SHORT,
    IPROT_SW_DATA_TOO_LONG,
    IPROT_SW_RQ_BAD_CRC,
    IPROT_SW_CTL_NOT_SUPPORTED,
    IPROT_SW_READ_ERROR,
    IPROT_SW_ID_NOT_EXIST,
    IPROT_SW_BAD_DST,
#if defined(__linux__) || defined(__APPLE__)
    KERN_STATUS_TIMEOUT = 0x80000001,
    KERN_STATUS_DEVICE_BUSY,
    KERN_STATUS_UNSUCCESSFUL,
    KERN_STATUS_INVALID_PARAMETER,
    KERN_STATUS_INVALID_DEVICE_STATE,
#endif
    // --- ExoDeviceAccess
    EDA_CANNOT_LOCALLY_OPEN_DRIVER = 0x30000000,
    EDA_INVALID_HND,
    EDA_CANNOT_OPEN_DRIVER,
    EDA_CANNOT_CLOSE_DRIVER,
    EDA_CANNOT_CLOSE_TLS_DRIVER_HANDLE,
    EDA_DRIVER_ALREADY_IN_USED,
    EDA_INVALID_MEM_ADDRESS_RANGE,
    EDA_INVALID_READ_RANGE,
    EDA_DATA_NOT_ENTIRELY_WRITTEN,
    EDA_INVALID_SUBSYSTEM_ID,
    EDA_INVALID_FIP_SPEED,
    EDA_INVALID_FIP_IMPEDANCE,
    EDA_IOCTL_CALL_ERROR,
    EDA_INVALID_OUT_AFTER_IOCTL_CALL,
    // --- KeyProtection
    FIPCORE_SOFTWARE_PROTECTION = 0x30001000,
    FIPCORE_SOFTWARE_TYPE,
    // --- FipCodeInstall
    FIPCORE_PERIODIC_CHANNEL_UNKNOWN = 0x30002000,
    FIPCORE_VARIABLE_LENGTH_ERROR,
    FIPCORE_REFRESH_TYPE_UNKNOWN,
    FIPCORE_VARIABLE_TYPE_UNKNOWN,
    FIPCORE_FULLFIP_ACCESS_ERROR,
    FIPCORE_WRONG_BA_FILE_FORMAT,
    FIPCORE_BA_INSTRUCTION_UNKNOWN,
    FIPCORE_CANNOT_OPEN_FIPCODE_FILE,
    FIPCORE_FIPCODE_CHECK_FAILED,
    FIPCORE_ILLEGAL_BA_PARAMETER,
    FIPCORE_TOO_MANY_FRAME_DESCRIPTOR,
    FIPCORE_NOT_ENOUGH_MEMORY_FOR_CONFIGURATION,
    FIPCORE_TOO_MANY_VARIABLE_DESCRIPTOR,
    FIPCORE_WRONG_FRAME_TYPE_IN_CONF_FILE,
    FIPCORE_NOT_ENOUGH_MEMORY_FOR_MSG_FIFO,
    FIPCORE_NOT_ENOUGH_MEMORY_FOR_BA,
    FIPCORE_NOT_ENOUGH_MEMORY_FOR_BA1,
    FIPCORE_NOT_ENOUGH_MEMORY_FOR_BA2,
    // --- FipCodeAccess
    FIPCORE_INVALID_FIP_USERCONF = 0x30003000,
    FIPCORE_INVALID_FIP_DEVICECONF,
    FIPCORE_BA_PROG_NUMBER_UNKNOWN,

    FIPCORE_INVALID_PAGE_NUMBER = 0x30004000,
    FIPCORE_PAGE_NOT_ENTIRELY_READ,
    FIPCORE_PAGE_NOT_ENTIRELY_WRITE,
    FIPCORE_INVALID_MEM_ADDRESS,
    FIPCORE_INVALID_ACK_EOC,
    FIPCORE_INVALID_CHANNEL_NUMBER,
    FIPCORE_SIZE16_AND_VAR_LENGTH_NOT_EQUAL,
    FIPCORE_NOT_VAR_TIME,
    FIPCORE_SOFTWARE_NOT_OPERATIONAL,
    FIPCORE_DATA_STORAGE,

    FIPCORE_NO_MEMORY_FOR_FIPCODEACCESS = 0x30005000,
    FIPCORE_NO_MEMORY_FOR_FIPCODECONF,
    FIPCORE_NO_MEMORY_FOR_FIPUSERCONF,
    FIPCORE_NO_MEMORY_FOR_FIPDEVICECONF,
    FIPCORE_NO_MEMORY_FOR_FIPCONTEXT,
    FIPCORE_NO_MEMORY_FOR_VARIABLE_DESCRIPTOR_TAB,
    FIPCORE_NO_ACCESSKEY_FOUND,
    FIPCORE_OPC_MEMORY_ALREADY_ALLOCATED,
    FIPCORE_NO_MEMORY_FOR_OPC_SERVER,
    FIPCORE_NO_ID_FOUND,
    // --- ParseConfFile
    FIPCORE_XML_PARSING_ERROR = 0x30006000,
    FIPCORE_XML_GET_ROOT_ELEMENT_ERROR,
    FIPCORE_XML_GET_ELEMENT_ERROR,
    FIPCORE_INVALID_XML_ROOT_ELEMENT,
    FIPCORE_INVALID_EXO_FILE_TYPE,
    // --- FipCoreMemMng
    FIPCORE_INIT_ERROR = 0x30007000,
    // --- FipCore
    FIPCORE_NOT_IMPLEMENTED_FOR_THIS_DEVICE_TYPE = 0x30008000,
    FIPCORE_FIFO_WRITE_TEST_ERROR,
    FIPCORE_FIFO_READ_TEST_ERROR,
    FIPCORE_IRQ_TEST_ERROR,
    FIPCORE_REGISTERS_TEST_ERROR,
    FIPCORE_TIMERS_TEST_ERROR,
    FIPCORE_INVALID_EVENT_DEFINITION,
    FIPCORE_NULL_POINTER,
    FIPCORE_INVALID_ACCESSRIGHTS,
    FIPCORE_CONF_NOT_FOUND,
    FIPCORE_TAG_TYPE_ID_UNKNOWN,
    FIPCORE_FILE_ALREADY_ALLOCATED,
    FIPCORE_TRANSFER_TIMEOUT,
    FIPCORE_TRANSFER_ALREADY_IN_PROGRESS,
    FIPCORE_TRANSFER_RETRIES_EXCEED_LIMIT,
    FIPCORE_RETRIEVING_FILE_NAME_ERROR,
    FIPCORE_TRANSFER_ABORTED_BY_USER,
    FIPCORE_FILE_RENAMING,
    FIPCORE_FILE_READING_ERROR,
    FIPCORE_FILE_WRITING_ERROR,
    FIPCORE_EOF_REACHED,
    FIPCORE_NOT_SUPPORT_FILE_TRANSFER,
    FIPCORE_CHUNKS_LOST,
    FIPCORE_IRQ_THREAD_ERROR,
    // --- FipCore - OPC UA
    FIPCORE_OPCUA_ALREADY_STARTED = 0x30009017,
    FIPCORE_OPCUA_ALREADY_STOPPED,
    FIPCORE_OPCUA_RUN_STARTUP,
    // --- CGenCard
    GENC_NO_ABSTRACTCOM_ASSOCIATED = 0x30100001,
    GENC_NO_REPLY_TO_READ_MEMORY_REQUEST,
    GENC_NO_REPLY_TO_READ_BY_ID_REQUEST,
    GENC_NO_REPLY_TO_READ_SN_REQUEST,
    GENC_NO_REPLY_TO_READ_LAST_SW_REQUEST,
    GENC_NO_REPLY_TO_READ_ERROR_REQUEST,
    GENC_NO_REPLY_TO_WRITE_MEMORY_REQUEST,
    GENC_NO_REPLY_TO_WRITE_BY_ID_REQUEST,
    GENC_NO_REPLY_TO_EXEC_BY_ID_REQUEST,
    GENC_NO_REPLY_TO_RESET_REQUEST,
    GENC_NO_REPLY_TO_SET_MODE_REQUEST,
    GENC_NO_REPLY_TO_GET_MODE_REQUEST,
    GENC_READ_DATA_NUMBER_NOT_MATCH,
    GENC_WAIT_FOR_REPLY_TIMEOUT,
    GENC_INCONSISTENT_FRAME,
    GENC_WAIT_FOR_REPLY_MORE_THAN_ONE_FRAME,
    GENC_NOT_IMPLEMENTED_FUNCTION,
    GENC_WAIT_FOR_EVENT_TIMEOUT,
    // --- CAbstractCom
    ABCOM_COMMUNICATION_UNOPENED = 0x30101000,
    // --- CUsbFtd2xxCom
    FTD2XX_CANNOT_LOAD_FTD2XX = 0x30102000,
    FTD2XX_CANNOT_READ_ALL_DATA_FROM_BUFFER,
    FTD2XX_CANNOT_WRITE_ALL_DATA_TO_BUFFER,
    FTD2XX_CANNOT_FIND_FTSETVIDPID,
    FTD2XX_CANNOT_FIND_FTGETVIDPID,
    FTD2XX_CANNOT_FIND_FTCREATEDEVICEINFOLIST,
    FTD2XX_CANNOT_FIND_FTGETDEVICEINFOLIST,
    FTD2XX_CANNOT_FIND_FTGETDEVICEINFODETAIL,
    FTD2XX_CANNOT_FIND_FTLISTDEVICES,
    FTD2XX_CANNOT_FIND_FTOPEN,
    FTD2XX_CANNOT_FIND_FTOPEN_EX,
    FTD2XX_CANNOT_FIND_FTCLOSE,
    FTD2XX_CANNOT_FIND_FTREAD,
    FTD2XX_CANNOT_FIND_FTWRITE,
    FTD2XX_CANNOT_FIND_FTSETBAUDRATE,
    FTD2XX_CANNOT_FIND_FTSETDIVISOR,
    FTD2XX_CANNOT_FIND_FTSETDATACHARACTERISTICS,
    FTD2XX_CANNOT_FIND_FTSETTIMEOUTS,
    FTD2XX_CANNOT_FIND_FTSETFLOWCONTROL,
    FTD2XX_CANNOT_FIND_FTSETDTR,
    FTD2XX_CANNOT_FIND_FTCLRDTR,
    FTD2XX_CANNOT_FIND_FTSETRTS,
    FTD2XX_CANNOT_FIND_FTCLRRTS,
    FTD2XX_CANNOT_FIND_FTGETMODEMSTATUS,
    FTD2XX_CANNOT_FIND_FTGETQUEUESTATUS,
    FTD2XX_CANNOT_FIND_FTGETDEVICEINFO,
    FTD2XX_CANNOT_FIND_FTGETDRIVERVERSION,
    FTD2XX_CANNOT_FIND_FTGETLIBRARYVERSION,
    FTD2XX_CANNOT_FIND_FTGETCOMPORTNUMBER,
    FTD2XX_CANNOT_FIND_FTGETSTATUS,
    FTD2XX_CANNOT_FIND_FTSETEVENTNOTIFICATION,
    FTD2XX_CANNOT_FIND_FTSETCHARS,
    FTD2XX_CANNOT_FIND_FTSETBREAKON,
    FTD2XX_CANNOT_FIND_FTSETBREAKOFF,
    FTD2XX_CANNOT_FIND_FTPURGE,
    FTD2XX_CANNOT_FIND_FTRESETCOM,
    FTD2XX_CANNOT_FIND_FTRESETPORT,
    FTD2XX_CANNOT_FIND_FTCYCLEPORT,
    FTD2XX_CANNOT_FIND_FTRESCAN,
    FTD2XX_CANNOT_FIND_FTRELOAD,
    FTD2XX_CANNOT_FIND_FTSETRESETPIPERETRYCOUNT,
    FTD2XX_CANNOT_FIND_FTSTOPINTASK,
    FTD2XX_CANNOT_FIND_FTRESTARTINTASK,
    FTD2XX_CANNOT_FIND_FTSETDEADMANTIMEOUT,
    FTD2XX_CANNOT_FIND_FTREADEE,
    FTD2XX_CANNOT_FIND_FTWRITEEE,
    FTD2XX_CANNOT_FIND_FTERASEEE,
    FTD2XX_CANNOT_FIND_FTEEREAD,
    FTD2XX_CANNOT_FIND_FTEEREADEX,
    FTD2XX_CANNOT_FIND_FTEEPROGRAM,
    FTD2XX_CANNOT_FIND_FTEEPROGRAMEX,
    FTD2XX_CANNOT_FIND_FTEEUASIZE,
    FTD2XX_CANNOT_FIND_FTEEUAREAD,
    FTD2XX_CANNOT_FIND_FTEEUAWRITE,
    FTD2XX_CANNOT_FIND_FTEEPROMREAD,
    FTD2XX_CANNOT_FIND_FTEEPROMPROGRAM,
    FTD2XX_CANNOT_FIND_FTSETLATENCYTIMER,
    FTD2XX_CANNOT_FIND_FTGETLATENCYTIMER,
    FTD2XX_CANNOT_FIND_FTSETBITMODE,
    FTD2XX_CANNOT_FIND_FTGETBITMODE,
    FTD2XX_CANNOT_FIND_FTSETUSBPARAMETERS,
    FTD2XX_INVALID_HANDLE,
    FTD2XX_DEVICE_NOT_FOUND,
    FTD2XX_DEVICE_NOT_OPENED,
    FTD2XX_IO_ERROR,
    FTD2XX_INSUFFICIENT_RESOURCES,
    FTD2XX_INVALID_PARAMETER,
    FTD2XX_INVALID_BAUD_RATE,
    FTD2XX_DEVICE_NOT_OPENED_FOR_ERASE,
    FTD2XX_DEVICE_NOT_OPENED_FOR_WRITE,
    FTD2XX_FAILED_TO_WRITE_DEVICE,
    FTD2XX_EEPROM_READ_FAILED,
    FTD2XX_EEPROM_WRITE_FAILED,
    FTD2XX_EEPROM_ERASE_FAILED,
    FTD2XX_EEPROM_NOT_PRESENT,
    FTD2XX_EEPROM_NOT_PROGRAMMED,
    FTD2XX_INVALID_ARGS,
    FTD2XX_NOT_SUPPORTED,
    FTD2XX_OTHER_ERROR,
    FTD2XX_DEVICE_LIST_NOT_READY,
    // --- CUsbCypCom
    CYP_INVALID_ENDPOINT = 0x30103000,
    CYP_IN_ENDPOINT_TIMEOUT,
    CYP_OUT_ENDPOINT_TIMEOUT,
    CYP_DEVICE_NOT_OPENED,
    // --- TcpIp
    TCPIP_WRITTING_ERROR = 0x30104000,
    // --- CUsbLibUsbCom
    LIBUSB_ERR_IO = 0x30105000,
    LIBUSB_ERR_INVALID_PARAM,
    LIBUSB_ERR_ACCESS,
    LIBUSB_ERR_NO_DEVICE,
    LIBUSB_ERR_NOT_FOUND,
    LIBUSB_ERR_BUSY,
    LIBUSB_ERR_TIMEOUT,
    LIBUSB_ERR_OVERFLOW,
    LIBUSB_ERR_PIPE,
    LIBUSB_ERR_INTERRUPTED,
    LIBUSB_ERR_NO_MEM,
    LIBUSB_ERR_NOT_SUPPORTED,
    LIBUSB_ERR_OTHER,
    // --- FipBusView
    FBV_NETWORK_NOT_STARTED = 0x30200001,
    // --- FipWatcherCard
    FWCARD_INVALID_OBJECT = 0x20100000,
    FWCARD_IS_ALREADY_LOCALLY_OPENED,
    FWCARD_IS_ALREADY_REMOTELY_OPENED,
    FWCARD_REMOTE_CONNECTION_TIMEOUT,
    FWCARD_REMOTE_DISCONNECTION_TIMEOUT,
    FWCARD_REMOTE_CONNECTION_INTERRUPTED,
    FWCARD_INVALID_INPUT_PARAMETERS,
    FWCARD_FCT_NOT_SUPPORTED_IN_REMOTE_CO,
    FWCARD_DEVICE_NOT_OPENED,
    FWCARD_REPLY_SIZE_ERROR
} EXO_ERROR_CODE;

typedef enum 
{ 
    FIPCORE,        // FipCore basic license 
    FIPCOREPLUS,    // FipCore extended license (FipBusView, OPC UA)
    FIPLABS,        // FipLabs basic license
} EXO_SOFT_TYPE;

typedef enum {
    XML = 0,
    CNF
} EXO_FILE_TYPE;

typedef enum
{ 
    PCI_PCIE_TYPE = 1,
    USB_TYPE,
    UNKNOWN_TYPE,
    MAX_TYPE = UNKNOWN_TYPE
} EXO_DEVICE_TYPE;

typedef enum
{ 
    EXO_SUBSYSID_OLD_PCI = 0x9050,
    // --- copper
    EXO_SUBSYSID_PCI = 0x5201,
    EXO_SUBSYSID_PCIE = 0x5301,
    EXO_SUBSYSID_PCI104 = 0x5401,
    EXO_SUBSYSID_PCIE104 = 0x5501,
    EXO_SUBSYSID_PXI = 0x5601,
    EXO_SUBSYSID_PXIE = 0x5701,
    EXO_SUBSYSID_MPCIE = 0x5801,
    // --- optical
    EXO_SUBSYSID_PCI_OPT = 0x7201,
    EXO_SUBSYSID_PCIE_OPT = 0x7301,
    EXO_SUBSYSID_PCI104_OPT = 0x7401,
    EXO_SUBSYSID_PCIE104_OPT = 0x7501,
    EXO_SUBSYSID_PXI_OPT = 0x7601,
    EXO_SUBSYSID_PXIE_OPT = 0x7701,
    EXO_SUBSYSID_MPCIE_OPT = 0x7801
} EXO_SUBSYSID;

typedef enum {
    SPEED_31K25 = 0,
    SPEED_1M,
    SPEED_2M5,
    SPEED_5M,
    UNKNOWN_SPEED
} FIP_SPEED;

typedef enum {
    IMPEDANCE_150 = 0,
    IMPEDANCE_120,
    UNKNOWN_IMPEDANCE
} FIP_IMPEDANCE;

typedef enum {
    FRAME_TYPE_FIP = 0,
    FRAME_TYPE_WORLDFIP,
    FRAME_TYPE_SLOWFIP,
    UNKNOWN_FRAME_TYPE
} FIP_FRAME_TYPE;

typedef enum {
    BLACK = 0,
    BLUE,
    GREEN,
    BLUE_SKY,
    RED,
    FUSCHIA,
    YELLOW,
    WHITE
} LED_RGB_COLOR;

typedef enum {
    MEDIUM_CMD_CLR_CHANNELS_ERR = 0x0010,
    MEDIUM_CMD_RST_CH1 = 0x0020,
    MEDIUM_CMD_RST_CH2 = 0x0040,
    MEDIUM_CMD_ACTIVATE_CH1 = 0x0009,
    MEDIUM_CMD_ACTIVATE_CH2 = 0x0006,
    MEDIUM_CMD_ACTIVATE_CH1_AND_CH2 = 0x0005
} MEDIUM_CMD;

/* bus arbiter status code - see FIP_NETWORK_STATUS struct - FIELDUAL_Status field */
typedef enum {
    MEDIUM_STATUS_CH1_VALID = (1 << 0),
    MEDIUM_STATUS_CH2_VALID = (1 << 1),
    MEDIUM_STATUS_CH1_TX_ERROR = (1 << 2),
    MEDIUM_STATUS_CH2_TX_ERROR = (1 << 3),
    MEDIUM_STATUS_CH1_WATCHDOG = (1 << 4),
    MEDIUM_STATUS_CH2_WATCHDOG = (1 << 5)
} MEDIUM_STATUS_FLAGS;

/* bus arbiter status code - see FIP_NETWORK_STATUS struct - BA_Status field */
typedef enum {
    /* ba stopped */
    BA_STATUS_STOPPED = 0x0000,
    /* ba in start-up phase */
    BA_STATUS_STARTING = 0x0003,
    /* ba in idle - another ba is already active */
    BA_STATUS_IDLE = 0x0007,
    /* end of macrocycle */
    BA_STATUS_MACRO_END = 0x8022,
    /* id_dat frame emission in progress */
    BA_STATUS_SENDING_ID_DAT = 0x8023,
    /* id_msg frame emission in progress */
    BA_STATUS_SENDING_ID_MSG = 0x8025,
    /* presence test in progress */
    BA_STATUS_TESTING = 0x8028,
    /* end of presence test */
    BA_STATUS_TESTING_END = 0x8029,
    /* ba in pending - wait for user action */
    BA_STATUS_PENDING = 0x8038,
    /* processing of aperiodic requests for variables */
    BA_STATUS_APER_WINDOW = 0x800B,
    /* processing of aperiodic requests for messages */
    BA_STATUS_MSG_WINDOW = 0x8004,
    /* internal re-synchronization */
    BA_STATUS_WAITING_TIME = 0x8010,
    /* external re-synchronization */
    BA_STATUS_WAITING_SYNC = 0x8031,
    /* external re-synchronization - silence */
    BA_STATUS_WAITING_SYNC_SILENT = 0x8032
} BA_STATUS_CODE;

/* fip network event code - see  fipcore_read_evt func - event_code parameter */
typedef enum {
    /* no event */
    FIP_EVT_NONE = 0X0000,
    /* sending of a variable - permanent */
    FIP_EVT_SEND_VAR_P = 0x8100,
    /* sending of a variable - temporary */
    FIP_EVT_SEND_VAR_T = 0x0100,
    /* reception of a variable - permanent */
    FIP_EVT_RECEIVE_VAR_P = 0x8200,
    /* reception of a variable - temporary */
    FIP_EVT_RECEIVE_VAR_T = 0x0200,
    /* reception of a message */
    FIP_EVT_RECEIVE_MSG = 0x0240,
    /* sending of a message */
    FIP_EVT_SEND_MSG = 0x0140,
    /* sending of a list of urgent aperiodic requests */
    FIP_EVT_SEND_APU = 0x0130,
    /* sending of a list of normal aperiodic requests */
    FIP_EVT_SEND_APN = 0x0131,
    /* bus arbiter reactivation (active on network) */
    FIP_EVT_BA_ACTIVITY = 0x0400,
    /* stopping of bus arbiter following timeout */
    FIP_EVT_BA_STOP1 = 0x0401,
    /* stopping of bus arbiter following network anomalies */
    FIP_EVT_BA_STOP2 = 0x0402,
    /* stopping of bus arbiter following user request */
    FIP_EVT_BA_STOP3 = 0x0404,
    /* return of bus arbiter to idle status */
    FIP_EVT_BA_IDLE = 0x0408,
    /* suspension of bus arbiter program on event number 1 */
    FIP_EVT_BA_SUSPEND0 = 0x0500,
    /* ... */
    FIP_EVT_BA_SUSPEND255 = 0x05ff
} FIP_EVENT_CODE;

typedef enum {
    IDLE = 0,
    IN_PROGRESS,
    COMPLETE,
    ABORTED
} FILE_TRANSFER_STATE;
#endif // _EXOCOMMON_H_

//### fipcore structures - concatenation of fipcodeaccesscommon file
#ifndef _FIPACCESSCOMMON_H_
#define _FIPACCESSCOMMON_H_

#if defined(_WIN32) || defined(_WIN64)
    typedef HANDLE      device_handle_t;
#else // defined(__linux__) || defined(__APPLE__)
    typedef intptr_t    device_handle_t;
#endif

/* opaque struct */
struct fipcore_device_handle;
typedef struct fipcore_device_handle fipcore_device_handle;

typedef uint32_t (*cb_file_completion)(fipcore_device_handle* hnd, uint8_t direction,
                    uint16_t dest_id, uint8_t dest_seg,
                    uint16_t src_id, uint8_t src_seg, uint32_t error, 
                    double transaction_duration_s, char* target_path, 
                    void* callback_ctx);
typedef void (*cb_file_progression)(fipcore_device_handle* hnd, uint8_t direction,
                    uint16_t dest_id, uint8_t dest_seg,
                    uint16_t src_id, uint8_t src_seg, uint32_t chunk_number,
                    uint32_t total_chunks, double time_elapsed_s, 
                    void* callback_ctx);
typedef void (*cb_fip_synchro)(fipcore_device_handle* hnd, void* callback_ctx);
typedef void (*cb_fip_event)(fipcore_device_handle* hnd, void* callback_ctx);

#pragma pack(1) // 1 byte alignment

//***************************************************************************************
//
//                              CONFIGURATION STRUCTURES
//
//***************************************************************************************
#define MAX_NAME_LENGTH 255
#define MAX_TYPE_LENGTH 255
#define MAX_UNIT_LENGTH 255
#define MAX_PATH_LENGTH 260

#define TAG_TYPES_BOOLEAN   0
#define TAG_TYPES_INT8      1
#define TAG_TYPES_UINT8     2
#define TAG_TYPES_INT16     3
#define TAG_TYPES_UINT16    4
#define TAG_TYPES_INT32     5
#define TAG_TYPES_UINT32    6
#define TAG_TYPES_INT64     7
#define TAG_TYPES_UINT64    8
#define TAG_TYPES_FLOAT     9
#define TAG_TYPES_DOUBLE    10
#define TAG_TYPES_ASCII     11
#define TAG_TYPES_HEXA      14

#define MAX_VAR_SIZE 128
#define MAX_MSG_SIZE 256
#define MSG_HEADER_SIZE 6

#define MAX_FIFO_EMISSION_BLOCKS 64

#define PDU_MPS 0x40
#define PDU_SM_MPS 0x50

#define MSG_MODE_NO_ACK 0x0000  /* message without ack */
#define MSG_MODE_ACK 0x0010     /* message with ack */

#pragma pack(1) // 1 byte alignment

typedef struct
{
    char op_code[20];

} INSTRUCTION;

typedef struct
{
    /* tag name */
    char        tag_name[MAX_NAME_LENGTH];
    /* tag id */
    uint16_t    tag_id;
    /* tag type name (ex: bool, int8, uint8 ...) */
    char        type_name[MAX_TYPE_LENGTH];
    /* tag type id (ex: TAG_TYPES_BOOLEAN) */
    uint16_t    type_id;
    /* access rights: 0: read/writeable, 1: readable */
    uint8_t     access_rights;
    /* tag description: value unit, comment ... */
    char        description[MAX_UNIT_LENGTH];
    /* fip variable identifier */
    uint16_t    identifier;
    /* start bit value (fip frame) */
    uint16_t    start_bit;
    /* stop bit value (fip frame) */
    uint16_t    stop_bit;
    /* endianness: -1: not significant, 0: big-endian, 1: little-endian */
    int8_t      endianness;
} USERDEF_TAG;

typedef struct
{
    /* fip variable name */
    char        name[MAX_NAME_LENGTH];
    /* fip variable identifier */
    uint16_t    identifier;
    /* pdu type: (0x40: mps pdu, 0x50: sm_mps pdu) */
    uint16_t    pdu;
    /* length in bytes */
    uint16_t    length;
    /* type: 0: produced, 1: consumed */
    uint8_t     type;
    /* refreshment: 0: no, 1: yes, 2: yes (dynamic) */
    uint8_t     refreshment_type;
    /* refreshment period */
    uint16_t    refreshment_period;
    /* promptness: 0: no, 1: yes */
    uint8_t     promptitude_type;
    /* promptness period */
    uint16_t    promptitude_period;
    /* message channel: -1: no channel, 0: aperiodic channel, 1..8: periodic channel */
    int16_t     msg_channel;
    /* messages service:
     * bit1=1   -> Messaging requests authorized
     * bit0=1   -> Message reception authorized
     * bit6=1   -> MPS aperiodic requests authorized
     */
    uint8_t     messages_service;
} USERDEF_VARIABLE; // (VARIABLE4)

typedef struct
{
    /* fip message name */
    char        name[MAX_NAME_LENGTH];
    // -- message characteristics
    /* type:
     * b7=1 -> message reserved for file transaction
     * b7=0 -> normal message
     * b0=1 -> message to transmit
     * b0=0 -> message to receive
     */
    uint8_t     type;
    /* message channel: 0: aperiodic channel, 1..8: periodic channel 
     * note: only significant for a message to transmit
     */
    uint8_t     channel;
    /* Acknowledgment request: 0: no, 1: yes
     * note: only significant for a message to transmit
     */
    uint8_t     acknowledgment;

    // -- header definition
    uint16_t    destination_identifier;
    uint8_t     destination_segment;
    uint16_t    source_identifier;
    uint8_t     source_segment;

} USERDEF_MESSAGE; // (MESSAGE4)

typedef struct
{
    uint8_t subscriber_number;
    char    subscriber_name[MAX_NAME_LENGTH];

} USERDEF_STATION_INF; // (INFO_STATION4)

typedef struct
{
    uint8_t         ba_max_subscriber_number;
    uint8_t         ba_priority;
    uint8_t         ba_max_priority;
    uint8_t         ba_optimization;
    uint16_t        nb_ba_1_instructions;
    INSTRUCTION*    ba_1_instructions;
    uint16_t        nb_ba_2_instructions;
    INSTRUCTION*    ba_2_instructions;

} USERDEF_ARBITER;

typedef struct
{
    USERDEF_STATION_INF* this_station;

    /* speed - 0: 31.25 kb/s, 1: 1 Mb/s, 2: 2.5 Mb/s, 3: 5 Mb/s */
    uint8_t             speed;
    /* tslot - 0, 1, 2 ou 3 */
    uint8_t             t_slot;
    /* silence_par (max: 0x3f) */
    uint8_t             silence;
    /* rptime_par (max: 0x3f) */
    uint8_t             turn_around;
    /* type - 0: FIP - 1: WorldFIP - 2: SlowFIP (fipcode5 1MHz compatibility) */
    uint8_t             frame_type;
    /* arbiter presence - 0: 0, 1: yes */
    bool                is_arbiter;
    /* fip arbitrator descriptor */
    USERDEF_ARBITER*    arbiter;
    /* number of fip mps variables */
    uint16_t            nb_variables;
    /* descriptors of the fip mps variables */
    USERDEF_VARIABLE*   variables;
    /* number of fip messages */
    uint16_t            nb_messages;
    /* descriptors of the fip messages */
    USERDEF_MESSAGE*    messages;
    /* number of fip tags */
    uint16_t            nb_tags;
    /* descriptors of the fip tags (for fip variables) */
    USERDEF_TAG*        tags;

} FIP_USERCONF; // User FIP Configuration (xml/cnf files)

typedef struct
{
    /* file completion callback */
    cb_file_completion  file_completion;
    /* file progression callback */
    cb_file_progression file_progression;
    /* synchro var 0x9003 callback (hardware irq) */
    cb_fip_synchro      fip_synchro;
    /* other fip event callback (hardware irq) */
    cb_fip_event        fip_event;
    /* callbacks context (ex: object pointer in cpp app) */
    void*               ctx_ptr;

} FIP_USERCALLBACKS; // User Callback Functions

typedef struct
{
    uint8_t             major;
    uint8_t             minor;
    uint8_t             revision;
} DEVICE_VERSION;

typedef struct
{
    /* device type to handle */
    EXO_DEVICE_TYPE     device_type;
    /* device index to handle */
    uint8_t             device_index;
    /* 1: device in use - 0: free device */
    uint8_t             device_is_used;
    /* handle on device driver */
    device_handle_t     device_handle;
    /* product number of the device */
    uint32_t            product_number;
    /* version number of the device */
    DEVICE_VERSION      device_version;
    /* fip speed */
    FIP_SPEED           fip_speed;
    /* fip impedance */
    FIP_IMPEDANCE       fip_impedance;
    /* subsystem id for pci/pcie devices */
    uint16_t            device_subsys_id;

#if defined(__linux__) || defined(__APPLE__) || defined(__QNX__)
    /* device general info */
    uint16_t            vid;
    uint16_t            did;
    uint16_t            ssvid;          // redundant with 'device_subsys_id'
    uint16_t            ssid;
    /* physical BAR addresses */
    unsigned long       plx_base;       // BAR 1 
    unsigned long       plx_size;       // BAR 1 size
    unsigned long       fullfip_base;   // BAR 2
    unsigned long       fullfip_size;   // BAR 2 size
    unsigned long       fieldual_base;  // BAR 3
    unsigned long       fieldual_size;  // BAR 3 size
    /* irq infos*/
    int                 irq_number;
#endif
} FIP_DEVICECONF;


//***************************************************************************************
//
//                              COMMUNICATION STRUCTURES  
//
//***************************************************************************************

//***************************************************************************************
// CONSUMED_VARIABLE
// Structure related to a consumed variable. - byte array [cf FIPCODE v6_4 p.112]
// <CONSUMED_VARIABLE> ::=      <Pdu><Length>{<Data_Var>}^Length-2
//                          or  <Pdu><Length>{<Data_Var>}^Length-3<Production_Status>
//                          if refreshment is active
// Used in: 
//  - FIP_READ_VAR
//  - FIP_READ_VAR_SVF
//  - USER_READ_VAR
//***************************************************************************************
typedef struct
{
    uint8_t Pdu;
    uint8_t Length;
    uint8_t Data_Var[MAX_VAR_SIZE];
    uint8_t Production_Status;

} CONSUMED_VARIABLE;

//***************************************************************************************
// CONSUMED_VARIABLE_TIME
// Structure related to a consumed variable with dynamic refreshment.
// byte array [cf FIPCODE v6_4 p.118].
// <CONSUMED_VARIABLE_TIME> ::= 
//              <Pdu><Length>{<Data_Var>}^Length-7<Production_Time><Production_Status>
//
// Used in:
//  - FIP_READ_VAR_TIME
//  - USER_READ_VAR_TIME
//***************************************************************************************
typedef struct
{
    uint8_t Pdu;
    uint8_t Length;
    uint8_t Data_Var[MAX_VAR_SIZE];
    uint32_t Production_Time;
    uint8_t Production_Status;

} CONSUMED_VARIABLE_TIME;

//***************************************************************************************
// PRODUCED_VARIABLE
// Structure related to a produced variable - byte array [cf FIPCODE v6_4 p.126]
// <PRODUCED_VARIABLE> ::=      {<Data_Var>}^n                      n = [6 ... 126]
//                          ou  {<Data_Var>}^n<Production_Status>   n = [1 ... 125]
// Note: PRODUCED_VARIABLE can be used as PRODUCED_VARIABLE_TIME. In this case,
// <Production_Time> and <Production_Status> must be inserted after useful data in Data_Var.
//
// Used in:
//  - FIP_WRITE_VAR
//  - USER_WRITE_VAR
//***************************************************************************************
typedef struct
{
    uint8_t Data_Var[MAX_VAR_SIZE - 2];

} PRODUCED_VARIABLE;

//***************************************************************************************
// PRODUCED_VARIABLE_TIME
// Structure related to a produced variable with dynamic refreshment.
// byte array [cf FIPCODE v6_4 p.126].
// <PRODUCED_VARIABLE_TIME> ::= 
//              {<Data_Var>}^n<Production_Time><Production_Status>  n = [6 ... 120]
//
// Used in:
//  - USER_WRITE_VAR_TIME
//***************************************************************************************
typedef struct
{
    uint8_t Data_Var[MAX_VAR_SIZE];
    uint8_t Production_Time;
    uint8_t Production_Status;

} PRODUCED_VARIABLE_TIME;

//***************************************************************************************
// PRESENT_LIST_BI_MEDIUM
// Structure containing the list of subscriber number (SM_MPS presence variable)
// for BI-MEDIUM.
// byte array [cf FIPCODE v6_4 p.122].
// Note: PRESENT_LIST is used as generic structure to contain bimedium or monomedium
// presence list.
// Example:
// Medium_Tabi[0].Bit0 = 1      -> Subscriber 0 is present on the medium i (i = 1 or 2)
// Medium_Tabi[0].Bit7 = 0      -> Subscriber 7 is absent on the medium i
// Medium_Tabi[31].Bit7 = 1     -> Subscriber 247 is present on the medium i
// Medium_Tabi[31].Bit7 = 1     -> Subscriber 255 is present on the medium i
//
// Used in:
//  - FIP_READ_LSTPRESENT
//  - USER_READ_LSTPRESENT
//***************************************************************************************
typedef struct
{
    uint8_t Pdu;                        // PDU_SM_MPS
    uint8_t Total_Length;               // 0x46
    uint8_t Medium_Tab1_Id;             // 0x80
    uint8_t Medium_Tab1_Length;         // 0x20     
    uint8_t Medium_Tab1[32];            // Presence list of the medium 1
    uint8_t Medium_Tab2_Id;             // 0x81
    uint8_t Medium_Tab2_Length;         // 0x20
    uint8_t Medium_Tab2[32];            // Presence list of the medium 2

} PRESENT_LIST_BI_MEDIUM, PRESENT_LIST;

//***************************************************************************************
// PRESENT_LIST_MONO_MEDIUM
// Structure containing the list of subscriber number (SM_MPS presence variable)
// for MONO-MEDIUM.
// byte array [cf FIPCODE v6_4 p.122].
//
// Medium_Tab[0].Bit0 = 1      -> Subscriber 0 is present on the medium
// Medium_Tab[0].Bit7 = 0      -> Subscriber 7 is absent on the medium
// Medium_Tab[31].Bit7 = 1     -> Subscriber 247 is present on the medium
// Medium_Tab[31].Bit7 = 1     -> Subscriber 255 is present on the medium
//
// Used in:
//  - FIP_READ_LSTPRESENT
//  - USER_READ_LSTPRESENT
//***************************************************************************************
typedef struct
{
    uint8_t Pdu;                        // PDU_SM_MPS
    uint8_t Total_Length;               // 0x46
    uint8_t Medium_Tab_Id;              // 0x80
    uint8_t Medium_Tab_Length;          // 0x20     
    uint8_t Medium_Tab[32];             // Presence list of the medium

} PRESENT_LIST_MONO_MEDIUM;

//***************************************************************************************
// MESSAGE_FRAME_OUT
// Structure defining the body of a message to be sent [cf FIPCODE v6_4 p.155]
//
// Used in:
//  - FIP_WRITE_MSG
//  - USER_WRITE_MSG
//***************************************************************************************

typedef struct
{
    // -- characteristics
    uint8_t Msg_Mode_PF;        // MSG_MODE_NO_ACK ou MSG_MODE_ACK
    uint8_t Msg_Mode_Pf;        // (respectively without/with ack)
    uint8_t Msg_Length_PF;      // Length = Header + payload data
    uint8_t Msg_Length_Pf;
    // -- message header
    uint8_t Dest_Add_PF;
    uint8_t Dest_Add_Pf;
    uint8_t Dest_Segment;
    uint8_t Src_Add_PF;
    uint8_t Src_Add_Pf;
    uint8_t Src_Segment;
    // -- payload data
    uint8_t Useful_Data[MAX_MSG_SIZE];

} MESSAGE_FRAME_OUT;

//***************************************************************************************
// MESSAGE_FRAME_IN
// Structure defining the body of a received message [cf FIPCODE v6_4 p.161]
//
// Used in:
//  - FIP_READ_MSG
//  - USER_READ_MSG
//***************************************************************************************
typedef struct
{
    // -- characteristics
    uint16_t Msg_Reception_Time;
    uint16_t Msg_Length;
    // -- message header
    uint8_t Dest_Add_PF;
    uint8_t Dest_Add_Pf;
    uint8_t Dest_Segment;
    uint8_t Src_Add_PF;
    uint8_t Src_Add_Pf;
    uint8_t Src_Segment;
    // -- payload data
    uint8_t Useful_Data[MAX_MSG_SIZE];

} MESSAGE_FRAME_IN;

//***************************************************************************************
// BA_PARAMETERS
// Structure of general parameters of the arbiter "on-line" editable
// [cf FIPCODE v6_4 p.192].
//
// Used in:
//  - FIP_CHANGE_PARAM_BA
//  - USER_CHANGE_PARAM_BA
//***************************************************************************************
typedef struct
{
    uint16_t BA_Subscriber_Number;
    uint16_t BA_Max_Subscriber_Number;
    uint16_t BA_Priority;
    uint16_t BA_Start_Time;
    uint16_t BA_Election_Time;

} BA_PARAMETERS;

//***************************************************************************************
// FIP_NETWORK_STATUS
// FIP Network Status Table - 10-word [cf FIPCODE v6_4 p.81]
//***************************************************************************************
typedef struct
{
    // -- channel counters
    uint16_t No_Error_Channel1;
    uint16_t No_Error_Channel2;
    uint16_t Error_Channel1;
    uint16_t Error_Channel2;
    uint16_t Error_No_Echo;
    uint16_t Error_Testp_Channel1;
    uint16_t Error_Testp_Channel2;
    // -- network status
    uint16_t BA_Current_Macrocyle;
    uint16_t BA_Status;
    uint16_t FIELDUAL_Status;

} FIP_NETWORK_STATUS;

//***************************************************************************************
// FIP_STATISTICS
// Extra statistics counters
//***************************************************************************************
typedef struct
{
    /*** Hardware IRQ/EOC handled or not */ 
    /* interrupt request produced by the fip device */
    uint32_t Irq_Handled_Count;
    /* interrupt request produced by another device (same IRQ line) */
    uint32_t Irq_Non_Handled_Count;

    /*** Extracted by driver from EOC Interruption */
    /* number of macrocycles - linked with fip synchro var (0x9003) */
    uint64_t Cycle_Count;
    /* last macrocycle time (period) - nanosecond resolution */
    uint64_t Cycle_Time;

    /*** Extracted by driver from IRQ Interruption */
    /* number of variables sent with a network event flag sets to "permanent" */
    uint64_t Var_P_Sent_Count;
    /* number of variables sent with a network event flag sets to "temporary" */
    uint64_t Var_T_Sent_Count;
    /* number of variables received with a network event flag sets to "permanent" */
    uint64_t Var_P_Received_Count;
    /* number of variables received with a network event flag sets to "temporary" */
    uint64_t Var_T_Received_Count;
    /* number of message received */
    uint64_t Msg_Received_Count;
    /* number of message sent */
    uint64_t Msg_Sent_Count;
    /* number of "urgent" requests for aperiodic variable list sent */
    uint64_t APU_List_Sent_Count;
    /* number of "normal" requests for aperiodic variable list sent */
    uint64_t APN_List_Sent_Count;
    /* counter of bus arbiter (re)activation */
    uint64_t BA_Activity_Count;
    /* counter of stopping of bus arbiter following timeout */
    uint64_t BA_Stop_On_Timeout_Count;
    /* counter of stopping of bus arbiter following network anomalies */
    uint64_t BA_Stop_On_Anomalies_Count;
    /* counter of stopping of bus arbiter following user request */
    uint64_t BA_Stop_By_User_Count;
    /* counter of return of bus arbiter to idle status */
    uint64_t BA_Idle_Count;
    /* counter of bus arbiter suspend */
    uint64_t BA_Suspend_Count;

} FIP_STATISTICS;

#pragma pack()

#endif // _FIPACCESSCOMMON_H_

//### fipcore functions
#ifdef __cplusplus
    extern "C" {
#endif

// --- Device Info Access
FIPCORE_EXPORT uint32_t fipcore_get_deviceconf(EXO_DEVICE_TYPE device_type,
                            uint8_t device_index, FIP_DEVICECONF* device_conf);
FIPCORE_EXPORT uint32_t fipcore_get_devices_number(uint8_t* nb_detected_devices);
FIPCORE_EXPORT uint32_t fipcore_get_devices_list(FIP_DEVICECONF* devices_conf,
                            uint8_t* nb_detected_devices);

// --- Device Access
FIPCORE_EXPORT fipcore_device_handle* fipcore_device_open(EXO_DEVICE_TYPE device_type,
                            uint8_t device_index, uint32_t* ret);
FIPCORE_EXPORT uint32_t fipcore_device_close(fipcore_device_handle* hnd);
FIPCORE_EXPORT uint32_t fipcore_get_rgb_led(fipcore_device_handle* hnd,
                            LED_RGB_COLOR* led_state);
FIPCORE_EXPORT uint32_t fipcore_set_rgb_led(fipcore_device_handle* hnd,
                            LED_RGB_COLOR led_state);
FIPCORE_EXPORT uint32_t fipcore_is_registered_device(fipcore_device_handle* hnd,
                            EXO_SOFT_TYPE soft_type);

// --- FIP Context Info
FIPCORE_EXPORT uint32_t fipcore_get_config(fipcore_device_handle* hnd,
                            FIP_USERCONF* fip_conf);
FIPCORE_EXPORT uint32_t fipcore_get_var_config_by_id(fipcore_device_handle* hnd,
                            uint16_t identifier, USERDEF_VARIABLE* var_conf);
FIPCORE_EXPORT uint32_t fipcore_get_var_key_by_id(fipcore_device_handle* hnd,
                            uint16_t identifier, uint8_t direction,
                            uint16_t* access_key);
FIPCORE_EXPORT uint32_t fipcore_get_var_id_by_key(fipcore_device_handle* hnd,
                            uint16_t access_key, uint16_t* identifier,
                            uint8_t* direction);
FIPCORE_EXPORT uint32_t fipcore_get_msg_config_by_header(fipcore_device_handle* hnd,
                            uint16_t dest_id, uint8_t dest_seg,
                            uint16_t src_id, uint8_t src_seg, USERDEF_MESSAGE* msg_conf);
FIPCORE_EXPORT uint32_t fipcore_get_tag_config_by_id(fipcore_device_handle* hnd,
                            uint16_t tag_id, USERDEF_TAG* tag_conf);
FIPCORE_EXPORT uint32_t fipcore_get_tag_config_by_name(fipcore_device_handle* hnd, 
                            const char* tag_name, USERDEF_TAG* tag_conf);

// --- System Management
FIPCORE_EXPORT uint32_t fipcore_init_network(fipcore_device_handle* hnd,
                            EXO_FILE_TYPE type, const char* file_conf_path, 
                            FIP_USERCALLBACKS* user_callbacks);
FIPCORE_EXPORT uint32_t fipcore_start_network(fipcore_device_handle* hnd);
FIPCORE_EXPORT uint32_t fipcore_stop_network(fipcore_device_handle* hnd);
FIPCORE_EXPORT uint32_t fipcore_get_medium(fipcore_device_handle* hnd, 
                            FIP_NETWORK_STATUS* fip_network_status);
FIPCORE_EXPORT uint32_t fipcore_set_medium_cmd(fipcore_device_handle* hnd,
                            uint16_t fieldual_cmd);

// --- Statistical Information
FIPCORE_EXPORT uint32_t fipcore_reset_stats(fipcore_device_handle* hnd);
FIPCORE_EXPORT uint32_t fipcore_get_stats(fipcore_device_handle* hnd, 
                            FIP_STATISTICS* stats);

// --- System Management - OPC UA Server
FIPCORE_EXPORT uint32_t fipcore_opcua_run_startup(fipcore_device_handle* hnd,
                            uint16_t port);
FIPCORE_EXPORT uint32_t fipcore_opcua_run_iterate(fipcore_device_handle* hnd,
                            bool wait_internal, uint16_t* timeout);
FIPCORE_EXPORT uint32_t fipcore_opcua_run_shutdown(fipcore_device_handle* hnd);
FIPCORE_EXPORT uint32_t fipcore_opcua_run(fipcore_device_handle* hnd, uint16_t port,
                            bool* running);

// --- Fip Variable Access
FIPCORE_EXPORT uint32_t fipcore_write_var_by_id(fipcore_device_handle* hnd,
                            uint16_t identifier, PRODUCED_VARIABLE* p_var);
FIPCORE_EXPORT uint32_t fipcore_read_var_by_id(fipcore_device_handle* hnd,
                            uint16_t identifier, CONSUMED_VARIABLE* c_var,
                            bool* is_prompt, bool* is_refreshed);
FIPCORE_EXPORT uint32_t fipcore_read_var_svf_by_id(fipcore_device_handle* hnd,
                            uint16_t identifier, CONSUMED_VARIABLE* c_var,
                            bool* is_prompt, bool* is_refreshed);
FIPCORE_EXPORT uint32_t fipcore_read_var_time_by_id(fipcore_device_handle* hnd,
                            uint16_t identifier, CONSUMED_VARIABLE_TIME* c_var_time,
                            uint32_t* production_time, uint32_t* transmission_time,
                            uint32_t* consumption_time, uint32_t* reception_time,
                            bool* is_prompt, bool* is_refreshed);
FIPCORE_EXPORT uint32_t fipcore_send_aper_by_id(fipcore_device_handle* hnd,
                            uint16_t identifier, uint8_t mode);
FIPCORE_EXPORT uint32_t fipcore_def_var_event_by_id(fipcore_device_handle* hnd,
                            uint16_t identifier, uint16_t event_code);
FIPCORE_EXPORT uint32_t fipcore_read_lstpresent(fipcore_device_handle* hnd,
                            PRESENT_LIST* present_list);
FIPCORE_EXPORT uint32_t fipcore_purge_aper(fipcore_device_handle* hnd,
                            uint8_t fifo_type);

// --- Fip Tag Access
FIPCORE_EXPORT uint32_t fipcore_write_tag_by_id(fipcore_device_handle* hnd,
                            uint16_t tag_id, void* data);
FIPCORE_EXPORT uint32_t fipcore_write_tag_by_name(fipcore_device_handle* hnd,
                            const char* tag_name, void* data);
FIPCORE_EXPORT uint32_t fipcore_read_tag_by_id(fipcore_device_handle* hnd,
                            uint16_t tag_id, void* data);
FIPCORE_EXPORT uint32_t fipcore_read_tag_by_name(fipcore_device_handle* hnd,
                            const char* tag_name, void* data);

// --- Fip Event Access
FIPCORE_EXPORT uint32_t fipcore_read_evt(fipcore_device_handle* hnd,
                            uint16_t* event_code, uint16_t* event_parameter);

// --- Fip Message Access
FIPCORE_EXPORT uint32_t fipcore_write_msg(fipcore_device_handle* hnd,
                            uint16_t channel_number,
                            MESSAGE_FRAME_OUT* message_frame_out);
FIPCORE_EXPORT uint32_t fipcore_read_msg(fipcore_device_handle* hnd,
                            MESSAGE_FRAME_IN* message_frame_in);
FIPCORE_EXPORT uint32_t fipcore_purge_msg(fipcore_device_handle* hnd,
                            uint16_t channel_number,
                            uint16_t* nb_blocks_freed, uint16_t* blocks_freed_tab);

// --- Fip Message Access - File Transfer
FIPCORE_EXPORT uint32_t fipcore_set_file_reception_dir_path(fipcore_device_handle* hnd,
                            const char* directory_path);
FIPCORE_EXPORT uint32_t fipcore_send_file(fipcore_device_handle* hnd,
                            uint16_t channel_number,
                            uint16_t dest_id, uint8_t dest_seg,
                            uint16_t src_id, uint8_t src_seg,
                            const char* file_path);
FIPCORE_EXPORT uint32_t fipcore_abort_file_emission(fipcore_device_handle* hnd);

// --- Fip Ba Access
FIPCORE_EXPORT uint32_t fipcore_start_ba(fipcore_device_handle* hnd, uint8_t ba_number);
FIPCORE_EXPORT uint32_t fipcore_stop_ba(fipcore_device_handle* hnd);

FIPCORE_EXPORT uint32_t fipcore_continue_ba(fipcore_device_handle* hnd,
                            uint16_t ba_continue_address);
FIPCORE_EXPORT uint32_t fipcore_change_macrocycle(fipcore_device_handle* hnd,
                            uint8_t ba_number);
FIPCORE_EXPORT uint32_t fipcore_synchro_ba(fipcore_device_handle* hnd);
FIPCORE_EXPORT uint32_t fipcore_change_param_ba(fipcore_device_handle* hnd,
                            BA_PARAMETERS* ba_parameters);

// --- Errors
FIPCORE_EXPORT char* fipcore_get_error_str(uint32_t error_code);

#ifdef __cplusplus
            }
#endif

#endif // _FIPCORE_H_
