//  Created on : 11 Feb 2016
//  Author : M. Coroyer
//  Company : © 2016 Exoligent
//  Description:
//      This is an example of using the FipCore library.
//      The xml stations files are built from fiplabs labpro files:
//          - 'file_transfer_31K25.labpro'  : 31.25 Kbps
//              -> sta0_file_transfer_31K25.xml
//              -> sta1_file_transfer_31K25.xml
//          - 'file_transfer_1M.labpro'     : 1     Mbps
//              -> sta0_file_transfer_1M.xml
//              -> sta1_file_transfer_1M.xml
//          - 'file_transfer_2M5.labpro'    : 2.5   Mbps
//              -> sta0_file_transfer_2M5.xml
//              -> sta1_file_transfer_2M5.xml
//          - 'file_transfer_5M.labpro'     : 5     Mbps
//              -> sta0_file_transfer_5M.xml
//              -> sta1_file_transfer_5M.xml
//
//  Station0 (sta0) Configuration
//      - Arbiter: true - 1 Scan Table
//          -> Scan Table 1:
//                  SEND_ID_DAT (0x8426)
//                  SEND_ID_DAT (0x8427)
//                  SEND_MSG (5 ms)
//                  SEND_APER (5 ms)
//                  NEXT_MACRO ()
//      - Periodic IDs
//          -> ID(0x8426) :
//                  Type:                               Out
//                  Length:                             12 bytes
//                  Refreshment:                        true (50ms)
//                  Aperiodic's List Transfert Request: false
//                  Message Emission Channel:           true (Type: Aperiodic - Channel: 0)
//                  Message Reception Channel:          true
//          -> ID(0x8427) :
//                  Type:                               In
//                  Promptness:                         true (50ms)
//                  Message Reception Channel:          false
//                  Other details: see Station1 (sta1) - ID(0x8427) 
//      - Aperiodic IDs
//          -> ID(0x8322) :
//                  Type:                               Out
//                  Length:                             10 bytes
//                  Refreshment:                        false
//                  Aperiodic's List Transfert Request: false
//                  Message Emission Channel:           false
//                  Message Reception Channel:          false
//      - Message(s)
//          -> ID (0x8426) to ID (0x8427)
//                  Source Address:                     Segment 0 - ID 0x8426
//                  Destination Address:                Segment 0 - ID 0x8427
//                  Acknowledgment:                     true
//                  Channel Emission Attached:          Type: Aperiodic - Channel: 0 (see ID(0x8426))
//                  File Transfer:                      true
//
//  Station1 (sta1) Configuration
//      - Arbiter: true - 1 Scan Table
//          -> Scan Table 1:
//                  SEND_ID_DAT (0x8426)
//                  SEND_ID_DAT (0x8427)
//                  SEND_MSG (5 ms)
//                  SEND_APER (5 ms)
//                  NEXT_MACRO ()
//      - Periodic IDs
//          -> ID(0x8426) :
//                  Type:                               In
//                  Promptness:                         true (50ms)
//                  Message Reception Channel:          false
//                  Other details: see Station0 (sta0) - ID(0x8426) 
//          -> ID(0x8427) :
//                  Type:                               Out
//                  Length:                             12 bytes
//                  Refreshment:                        true (200 ms)
//                  Aperiodic's List Transfert Request: false
//                  Message Emission Channel:           true (Type: Aperiodic - Channel: 0)
//                  Message Reception Channel:          true
//      - Message(s)
//          -> ID (0x8427) to ID (0x8426)
//                  Source Address:                     Segment 0 - ID 0x8427
//                  Destination Address:                Segment 0 - ID 0x8426
//                  Acknowledgment:                     true
//                  Channel Emission Attached:          Type: Aperiodic - Channel: 0 (see ID(0x8426))
//                  File Transfer:                      true
//
//  In this example:
//      station0_process routine: This station is used to the file emission
//          Open Device / Load Configuration / Run Bus Arbiter
//          On key press, the file is sent
//          Loop
//              Each 2 ms 
//                  Events are popped
//          End Loop on key press
//
//      station1_process routine: This station is used to the file reception
//          Open Device / Load Configuration / Run Bus Arbiter
//          Loop
//              Each 2 ms
//                  Events are popped
//          End Loop on key press

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "fipcore.h"

#if defined(_WIN32) || defined(_WIN64)
#   include <windows.h>
#   include <conio.h>
#elif defined(__linux__) || defined(__APPLE__) || defined(__QNX__)
#   include <unistd.h>     // usleep()
#   include <termios.h>
#   include <sys/types.h>
#   include <sys/time.h>
#   include <pthread.h>

void changemode(int dir)
{
    static struct termios oldt, newt;
    
    if (dir == 1) {
        tcgetattr(STDIN_FILENO, &oldt);
        newt = oldt;
        newt.c_lflag &= ~(ICANON | ECHO);
        tcsetattr(STDIN_FILENO, TCSANOW, &newt);
    } else
        tcsetattr(STDIN_FILENO, TCSANOW, &oldt);
}

int kbhit(void)
{
    struct timeval tv;
    fd_set rdfs;
    
    tv.tv_sec = 0;
    tv.tv_usec = 0;
    
    FD_ZERO(&rdfs);
    FD_SET(STDIN_FILENO, &rdfs);
    
    select(STDIN_FILENO + 1, &rdfs, NULL, NULL, &tv);
    return FD_ISSET(STDIN_FILENO, &rdfs);
}
#endif

uint16_t event_parameter = 0;

char* sta_xml_path[2][4] = {
    {"../xml/sta0_file_transfer_31K25.xml", "../xml/sta0_file_transfer_1M.xml", "../xml/sta0_file_transfer_2M5.xml", "../xml/sta0_file_transfer_5M.xml"},
    {"../xml/sta1_file_transfer_31K25.xml", "../xml/sta1_file_transfer_1M.xml", "../xml/sta1_file_transfer_2M5.xml", "../xml/sta1_file_transfer_5M.xml"}
};

void station0_process(uint8_t device_type, uint8_t device_index, char* xml_path, char* output_file_path);
void station1_process(uint8_t device_type, uint8_t device_index, char* xml_path, char* input_file_path);
void print_device_info(FIP_DEVICECONF* device_conf);
void event_processing();

uint32_t file_completion(fipcore_device_handle* hnd, uint8_t direction, 
            uint16_t dest_id, uint8_t dest_seg, uint16_t src_id, uint8_t src_seg,
            uint32_t tr_ret, double transaction_duration_s, char* path, void* callback_ctx) {
    uint32_t ret = SUCCESS;
    if (direction == 1) // Output File
        printf("\n[CALLBACK - FILE EMISSION END] From path: %s - Transfer duration %f s - Msg Vector: Src[ID: 0x%04x, Segment: %d] Dst[ID: 0x%04x, Segment: %d] - ",
            path, transaction_duration_s, src_id, src_seg, dest_id, dest_seg);
    else                // Input File
        printf("\n[CALLBACK - FILE RECEPTION END] To path: %s - Transfer duration %f s - Msg Vector: Src[ID: 0x%04x, Segment: %d] Dst[ID: 0x%04x, Segment: %d] - ",
            path, transaction_duration_s, src_id, src_seg, dest_id, dest_seg);
    printf("Report: %s\n", fipcore_get_error_str(tr_ret));
    return ret;
}

int main(int argCount, char *argValue[])
{
    uint32_t ret = SUCCESS;
    uint8_t i = 0;
    uint8_t device_type;
    uint8_t device_index;
    uint8_t station_number;
    uint8_t station_speed;
    FIP_DEVICECONF devices_list[20];
    uint8_t nb_detected_devices = 0;
    bool    is_matched = false;
    char    xml_path[255];
    char    temp[10];
    char    path[1024] = {0};

    // Device Enumeration
    printf("/*****************************************************************************/\n");
    printf("/*                          DEVICES ON COMPUTER                              */\n");
    printf("/*****************************************************************************/\n");

    ret = fipcore_get_devices_number(&nb_detected_devices);
    if (ret != SUCCESS) {
        printf("fipcore_get_devices_number - %s\n", fipcore_get_error_str(ret));
        return 0;
    }

    printf("NUMBER OF DETECTED DEVICES - %d\n", nb_detected_devices);

    ret = fipcore_get_devices_list(devices_list, &nb_detected_devices);
    if (ret != SUCCESS) {
        printf("fipcore_get_devices_list - %s\n", fipcore_get_error_str(ret));
        return 0;
    }

    for (i = 0; i < nb_detected_devices; i++)
        print_device_info(&(devices_list[i]));

    if (argCount < 5) {
        printf("This application must have this form: fipcore_file_transfer.exe 'device_type' 'device_index' 'station_number' 'station_speed' 'path'\n");
        printf("'device_type': 1: PCI_PCIE_TYPE - 2: USB_TYPE\n");
        printf("'device_index': 0 to 255\n");
        printf("'station_number': 0: Station 0 - 1: Station 1\n");
        printf("'station_speed': 0: 31.25 Kbps - 1: 1 Mbps - 2: 2.5 Mbps - 3: 5Mbps\n");
        printf("'path': if Station 0: file path to transfer - if Station 1: directory path to save (create)\n");
        return 0;
    }

    sscanf (argValue[1], "%s", temp);
    device_type = (uint8_t) atoi(temp);
    memset(temp, 0, sizeof(temp));

    sscanf (argValue[2], "%s", temp);
    device_index = (uint8_t) atoi(temp);
    memset(temp, 0, sizeof(temp));

    sscanf (argValue[3], "%s", temp);
    station_number = (uint8_t) atoi(temp);
    memset(temp, 0, sizeof(temp));

    sscanf (argValue[4], "%s", temp);
    station_speed = (uint8_t) atoi(temp);
    memset(temp, 0, sizeof(temp));

    sscanf (argValue[5], "%s", path);
    
    if (!(device_type < MAX_TYPE)) {
        printf("Invalid 'device_type' parameter.\n");
        return 0;
    }
    if (station_number > 1) {
        printf("Invalid 'station_number' parameter.\n");
        return 0; 
    }
    if (station_speed > 3) {
        printf("Invalid 'station_speed' parameter.\n");
        return 0; 
    }

    for (i = 0; i < nb_detected_devices; i++) {

        if ((devices_list[i].device_type == device_type) && 
            (devices_list[i].device_index == device_index)) {

                if (devices_list[i].fip_speed != station_speed) {

                    if (!((devices_list[i].fip_speed == SPEED_2M5) && 
                        (station_speed == SPEED_5M))) {
                        // Support 2.5 Mbps device with 5 Mbps configuration

                        // else exit
                        printf("Station speed must match with device speed.\n");
                        return 0;
                    }
                }
                is_matched = true;
        }
    }

    if (is_matched) {

        if (station_number == 0)
            station0_process(device_type, device_index, sta_xml_path[station_number][station_speed], path);
        else
            station1_process(device_type, device_index, sta_xml_path[station_number][station_speed], path);
    } else
        printf("No device was found with device_type (%d) and device_index (%d).\n", device_type, device_index);

    return 0;
}

void station0_process(uint8_t device_type, uint8_t device_index, char* xml_path, char* file_path)
{
    uint32_t ret = SUCCESS;
    PRODUCED_VARIABLE p_var = {0};
    FIP_USERCALLBACKS user_cb = {0};
    uint16_t src_id;
    uint16_t dst_id;
    fipcore_device_handle* sta0_hnd = NULL;

    printf("/*****************************************************************************/\n");
    printf("/*                          STATION 0 TEST                                   */\n");
    printf("/*****************************************************************************/\n");

    if (!file_path)
        goto _exit1;

    sta0_hnd = fipcore_device_open(device_type, device_index, &ret);
    if (ret != SUCCESS) {
        printf("STATION 0 - %s\n", fipcore_get_error_str(ret));
        goto _exit1;
    }

    
    user_cb.file_completion = &file_completion;
    ret = fipcore_init_network(sta0_hnd, XML, xml_path, &user_cb);
    if (ret != SUCCESS)
        goto _exit;

    ret = fipcore_start_network(sta0_hnd);
    if (ret != SUCCESS)
        goto _exit;

    ret = fipcore_start_ba(sta0_hnd, 1);
    if (ret != SUCCESS)
        goto _exit;

    // Output variables init
    fipcore_write_var_by_id(sta0_hnd, 0x8426, &p_var);

    // Output message init
    src_id = 0x8426;
    dst_id = 0x8427;

#if defined(__linux__) || defined(__APPLE__)
    changemode(1);
#endif

    printf("Press a key to start file transfer ...\n");
    getchar();

    ret = fipcore_send_file(sta0_hnd, 0, dst_id, 0, src_id, 0, file_path);
    if (ret != SUCCESS)
        goto _exit;

    while (!kbhit()) {

        event_processing(sta0_hnd); 
        
#if defined(_WIN32) || defined(_WIN64)
        Sleep(2);
#else
        usleep(2000);
#endif
    }

#if defined(__linux__) || defined(__APPLE__)
    changemode(0);
#endif
    
    ret = fipcore_stop_ba(sta0_hnd);

_exit:  
    if (ret != SUCCESS)
        printf("STATION 0 - %s\n", fipcore_get_error_str(ret));

    ret = fipcore_stop_network(sta0_hnd);
    if (ret != SUCCESS)
        printf("STATION 0 - %s\n", fipcore_get_error_str(ret));
_exit1:
    ret = fipcore_device_close(sta0_hnd);
    if (ret != SUCCESS)
        printf("STATION 0 - %s\n", fipcore_get_error_str(ret));
}

void station1_process(uint8_t device_type, uint8_t device_index, char* xml_path, char* directory_path)
{
    uint32_t ret = SUCCESS;
    PRODUCED_VARIABLE p_var = {0};
    FIP_USERCALLBACKS user_cb = {0};
    uint8_t i, j = 0;
    fipcore_device_handle* sta1_hnd = NULL;

#if defined(__linux__)
    int policy;
    struct sched_param param;
#endif

    printf("/*****************************************************************************/\n");
    printf("/*                          STATION 1 TEST                                   */\n");
    printf("/*****************************************************************************/\n");

    sta1_hnd = fipcore_device_open(device_type, device_index, &ret);
    if (ret != SUCCESS) {
        printf("STATION 1 - %s\n", fipcore_get_error_str(ret));
        goto _exit1;
    }

    user_cb.file_completion = &file_completion;
    ret = fipcore_init_network(sta1_hnd, XML, xml_path, &user_cb);
    if (ret != SUCCESS)
        goto _exit;

    ret = fipcore_set_file_reception_dir_path(sta1_hnd, directory_path);
    if (ret != SUCCESS)
        goto _exit;

    ret = fipcore_start_network(sta1_hnd);
    if (ret != SUCCESS)
        goto _exit;

    ret = fipcore_start_ba(sta1_hnd, 1);
    if (ret != SUCCESS)
        goto _exit;

    // Output variables init
    fipcore_write_var_by_id(sta1_hnd, 0x8427, &p_var);

#if defined(__linux__) || defined(__APPLE__)
    changemode(1);
#endif

    // The priority is increased to ensure that the receiving FIFO is emptied fairly quickly
#if defined(_WIN32) || defined(_WIN64)
    SetThreadPriority(GetCurrentThread(), THREAD_PRIORITY_HIGHEST);
#elif defined(__linux__)
    pthread_getschedparam(pthread_self(), &policy, &param);
    param.sched_priority = sched_get_priority_max(policy);
    pthread_setschedparam(pthread_self(), policy, &param);
#endif

    while (!kbhit()) {

        event_processing(sta1_hnd); 
        
#if defined(_WIN32) || defined(_WIN64)
        Sleep(2);
#else
        usleep(2000);
#endif
    }

#if defined(__linux__) || defined(__APPLE__)
    changemode(0);
#endif
    
    ret = fipcore_stop_ba(sta1_hnd);

_exit:  
    if (ret != SUCCESS)
        printf("STATION 1 - %s\n", fipcore_get_error_str(ret));

    ret = fipcore_stop_network(sta1_hnd);
    if (ret != SUCCESS)
        printf("STATION 1 - %s\n", fipcore_get_error_str(ret));
_exit1:
    ret = fipcore_device_close(sta1_hnd);
    if (ret != SUCCESS)
        printf("STATION 1 - %s\n", fipcore_get_error_str(ret));
}

void event_processing(fipcore_device_handle* hnd)
{
    uint32_t ret = SUCCESS;
    uint16_t event_code = 0;
    
    ret = fipcore_read_evt(hnd, &event_code, &event_parameter);

    if (ret == SUCCESS) {

        bool msgReceivedToTreat = false;

        //--printf("  fipcore_read_evt - ");
        switch (event_code) {
        case FIP_EVT_SEND_VAR_P:    printf("SEND_VAR_P - AccessKey: 0x%04x\n", event_parameter); break;
        case FIP_EVT_SEND_VAR_T:    printf("SEND_VAR_T - AccessKey: 0x%04x\n", event_parameter); break;
        case FIP_EVT_RECEIVE_VAR_P: printf("RECEIVE_VAR_P\n"); break;
        case FIP_EVT_RECEIVE_VAR_T: printf("RECEIVE_VAR_T\n"); break;
        case FIP_EVT_RECEIVE_MSG:   
            {
                msgReceivedToTreat = true;
                printf("|");
            }
        break;
        case FIP_EVT_SEND_MSG:
            {
                printf("|");
            } 
        break;
        case FIP_EVT_SEND_APU:      printf("SEND_APU\n"); break;
        case FIP_EVT_SEND_APN:      printf("SEND_APN\n"); break;
        case FIP_EVT_BA_ACTIVITY:   printf("BA_ACTIVITY\n"); break;
        case FIP_EVT_BA_STOP1:      printf("BA_STOP1\n"); break;
        case FIP_EVT_BA_STOP2:      printf("BA_STOP2\n"); break;
        case FIP_EVT_BA_STOP3:      printf("BA_STOP3\n"); break;
        case FIP_EVT_BA_IDLE:       printf("BA_IDLE\n"); break;
        default:
            if ((event_code & 0x0500) == 0x0500) {
                uint8_t suspend_number = (uint8_t) (event_code & 0x00FF);
                printf("BA_SUSPEND%d - BA Address Program: 0x%04x\n", suspend_number, event_parameter);
            } else
                printf("UNKNOWN - EventCode: 0x%04x - EventParameter: 0x%04x\n", event_code, event_parameter);          

        break;
        }

        if (msgReceivedToTreat) {
            MESSAGE_FRAME_IN c_msg = {0};

            ret = fipcore_read_msg(hnd, &c_msg); 
            if (ret != SUCCESS)
                printf("ERROR - %s\n", fipcore_get_error_str(ret));
        }

    //     ret = fipcore_read_evt(&event_code, &event_parameter);
    }

    if ((ret != SUCCESS) && (ret != FIPCODE_CR_FIFO_EVT_EMPTY))
        printf("ERROR - %s\n", fipcore_get_error_str(ret));
}

void print_device_info(FIP_DEVICECONF* device_conf)
{
    switch(device_conf->device_type) {
    case PCI_PCIE_TYPE:     printf("   device_type     : PCI_PCIE_TYPE \n"); break;
    case USB_TYPE:          printf("   device_type     : USB_TYPE \n"); break;
    default:                printf("   device_type     : UNKNOWN_TYPE \n"); break;
    }
    printf("   device_index    : %d \n", device_conf->device_index);
    printf("   product_number  : 0x%08x \n", device_conf->product_number);
    printf("   device_version  : %d.%d.%d \n", device_conf->device_version.major,
                                            device_conf->device_version.minor,
                                            device_conf->device_version.revision);
    switch(device_conf->fip_speed) {
    case SPEED_31K25:   printf("   fip_speed       : SPEED_31K25 \n"); break;
    case SPEED_1M:      printf("   fip_speed       : SPEED_1M \n"); break;
    case SPEED_2M5:     printf("   fip_speed       : SPEED_2M5 or SPEED_5M \n"); break;
    default:            printf("   fip_speed       : UNKNOWN_SPEED \n"); break;
    }
    switch(device_conf->fip_impedance) {
    case IMPEDANCE_150: printf("   fip_impedance   : IMPEDANCE_150 \n\n"); break;
    case IMPEDANCE_120: printf("   fip_impedance   : IMPEDANCE_120 \n\n"); break;
    default:            printf("   fip_impedance   : UNKNOWN_IMPEDANCE \n\n"); break;
    }
}