//  Created on : 4 Jan 2018
//  Author : M. Coroyer
//  Company : © 2014-2018 Exoligent
//  Description:
//  This example is similar to the "fipcore_mnemo" example
//  So for more information about the FIP configuration, please refer to "fipcore_mnemo"
//  Here, the tags "TA.PumpTemperature" and "TB.PumpTemperature" are no longer generated
//  randomly; we use a PT100 RTD sensor plugged to an ADAM-5000 Advantech module
//  (RTD module: ADAM-5013)
//  The communication with ADAM module is carried out via a serial link (RS-232)

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "fipcore.h"

#if defined(_WIN32) || defined(_WIN64)
#   include <windows.h>
#   include <conio.h>
#   include <time.h>
#elif defined(__linux__)
#   include <unistd.h>     // usleep()
#   include <termios.h>
#   include <sys/types.h>
#   include <sys/time.h>
#   include <fcntl.h>      // open()

int kbhit(void)
{
    struct timeval tv;
    fd_set rdfs;
    
    tv.tv_sec = 0;
    tv.tv_usec = 0;
    
    FD_ZERO(&rdfs);
    FD_SET(STDIN_FILENO, &rdfs);
    
    select(STDIN_FILENO + 1, &rdfs, NULL, NULL, &tv);
    return FD_ISSET(STDIN_FILENO, &rdfs);
}

unsigned GetTickCount()
{
    struct timeval tv;
    if(gettimeofday(&tv, NULL) != 0)
            return 0;

    return (tv.tv_sec * 1000) + (tv.tv_usec / 1000);
}
#endif

// ADAM CONSTANTS
#define ADAM5000_PORT_COM   1 /* > 0 */
#define ADAM5000_ADDR       1 /* range: 0-255 */
#define ADAM5013_SLOT_NUM   2 /* range: 0-3 */
#define ADAM5013_CHANNEL    0 /* range: 0-2 */
#if defined(_WIN32) || defined(_WIN64)
    HANDLE com_hnd = 0;

    COMMTIMEOUTS g_cto =
    {
        500,    /* ReadIntervalTimeOut          */
        0,      /* ReadTotalTimeOutMultiplier   */
        500,    /* ReadTotalTimeOutConstant     */
        0,      /* WriteTotalTimeOutMultiplier  */
        0       /* WriteTotalTimeOutConstant    */
    };

#elif defined(__linux__)
    int com_hnd = 0;
    struct termios oldtio;
#endif

// GLOBAL VARS
uint16_t event_parameter = 0;

char* sta_xml_path[2][4] = {
    {"../xml/sta0_worldfip_31K25.xml", "../xml/sta0_worldfip_1M.xml", "../xml/sta0_worldfip_2M5.xml", "../xml/sta0_worldfip_5M.xml"},
    {"../xml/sta1_worldfip_31K25.xml", "../xml/sta1_worldfip_1M.xml", "../xml/sta1_worldfip_2M5.xml", "../xml/sta1_worldfip_5M.xml"}
};

void station0_process(uint8_t device_type, uint8_t device_index, char* xml_path);
void station1_process(uint8_t device_type, uint8_t device_index, char* xml_path);
void print_device_info(FIP_DEVICECONF* device_conf);
void print_tag_conf(const USERDEF_TAG* ut);
void print_msg_conf(const USERDEF_MESSAGE* um);
void print_conf(const FIP_USERCONF* fuc);
void event_processing(fipcore_device_handle* hnd);
void medium_processing(fipcore_device_handle* hnd);

int adam_connect()
{
    char dev_name[100] = {0};

#if defined(_WIN32) || defined(_WIN64)

    sprintf(dev_name, "COM%d", ADAM5000_PORT_COM);
    com_hnd = CreateFile( dev_name,
                        GENERIC_READ | GENERIC_WRITE,
                        0,
                        NULL,
                        OPEN_EXISTING,
                        FILE_ATTRIBUTE_SYSTEM,
                        NULL);

    if (com_hnd == INVALID_HANDLE_VALUE)
        return -1;

    SetCommTimeouts(com_hnd, &g_cto);

#elif defined(__linux__)

    struct termios newtio;

    memset (&newtio, 0, sizeof(newtio));

    sprintf(dev_name, "/dev/ttyS%d", ADAM5000_PORT_COM);
    com_hnd = open(dev_name, O_RDWR);
    if (com_hnd == -1)
        return -1;

    fcntl(com_hnd, F_SETFL, 0); // blocking mode

    // configuration
    tcgetattr(com_hnd, &newtio);

    oldtio = newtio;

    // raw mode, no canonic mode, no echo
    newtio.c_iflag = IGNBRK;
    newtio.c_lflag = 0;
    newtio.c_oflag = 0;

    // RTS/CTS: yes
    newtio.c_cflag |= (CREAD | CRTSCTS);

    // 1 byte min
    newtio.c_cc[VMIN] = 1;

    // data available now
    newtio.c_cc[VTIME] = 0;

    // XON/XOFF: no
    newtio.c_iflag &= ~(IXON | IXOFF | IXANY);

    // 8n1
    newtio.c_cflag &= ~(PARENB | CSIZE);
    newtio.c_cflag |= CS8;

    // modem signal management
    newtio.c_cflag &= ~CLOCAL;

    tcsetattr(com_hnd, TCSANOW, &newtio);
    sleep(1);
#endif
    return 0;
}

void adam_disconnect()
{
#if defined(_WIN32) || defined(_WIN64)
    CloseHandle(com_hnd);
#elif defined(__linux__)
    tcsetattr(com_hnd, TCSANOW, &oldtio); // restore
    close(com_hnd);
#endif
}

float getTemperatureFloat()
{
    char cmd[10] = {0};
    char result[100] = {0};
    int bytes_read = 0;
    int bytes_written = 0;
    float temperature = 0;

    // To get rtd input from ADAM-5000, the cmd to send via com port is:
    // #aaSiCj where aa: adam addr - Si: slot i - Cj: channel j
    sprintf(cmd, "#%02xS%dC%d\r", ADAM5000_ADDR, ADAM5013_SLOT_NUM, ADAM5013_CHANNEL);

#if defined(_WIN32) || defined(_WIN64)

    WriteFile(com_hnd, cmd, strlen(cmd), (LPDWORD) &bytes_written, NULL);
    while (!bytes_read) 
        ReadFile(com_hnd, result, 100, (LPDWORD) &bytes_read, NULL);

#elif defined(__linux__)
    bytes_written += write(com_hnd, cmd, strlen(cmd));
    bytes_read = read(com_hnd, result, 100);
#endif

    if (result[0] == '?')
        printf("ADAM-5000: invalid cmd\n");
    else if (result[0] == '>') {
        // the operation is valid
        int sign = 1;
        if (result[1] == '-')
            sign = -1;

        temperature = (float) sign * atof(&result[2]);
    }

    return temperature;
}

int main(int argCount, char *argValue[])
{
    uint32_t ret = SUCCESS;
    uint8_t i = 0;
    uint8_t device_type;
    uint8_t device_index;
    uint8_t station_number;
    uint8_t station_speed;
    FIP_DEVICECONF devices_list[20];
    uint8_t nb_detected_devices = 0;
    bool    is_matched = false;
    char    xml_path[255];
    char    temp[10];

    // Device Enumeration
    printf("/*****************************************************************************/\n");
    printf("/*                          DEVICES ON COMPUTER                              */\n");
    printf("/*****************************************************************************/\n");

    ret = fipcore_get_devices_number(&nb_detected_devices);
    if (ret != SUCCESS) {
        printf("fipcore_get_devices_number - %s\n", fipcore_get_error_str(ret));
        return 0;
    }

    printf("NUMBER OF DETECTED DEVICES - %d\n", nb_detected_devices);

    ret = fipcore_get_devices_list(devices_list, &nb_detected_devices);
    if (ret != SUCCESS) {
        printf("fipcore_get_devices_list - %s\n", fipcore_get_error_str(ret));
        return 0;
    }

    for (i = 0; i < nb_detected_devices; i++)
        print_device_info(&(devices_list[i]));

    if (argCount < 4) {
        printf("This application must have this form: fipcore_pt100.exe 'device_type' 'device_index' 'station_number' 'station_speed' \n");
        printf("'device_type': 1: PCI_PCIE_TYPE - 2: USB_TYPE\n");
        printf("'device_index': 0 to 255\n");
        printf("'station_number': 0: Station 0 - 1: Station 1\n");
        printf("'station_speed': 0: 31.25 Kbps - 1: 1 Mbps - 2: 2.5 Mbps - 3: 5Mbps\n");
        return 0;
    }

    sscanf (argValue[1], "%s", temp);
    device_type = (uint8_t) atoi(temp);
    memset(temp, 0, sizeof(temp));

    sscanf (argValue[2], "%s", temp);
    device_index = (uint8_t) atoi(temp);
    memset(temp, 0, sizeof(temp));

    sscanf (argValue[3], "%s", temp);
    station_number = (uint8_t) atoi(temp);
    memset(temp, 0, sizeof(temp));

    sscanf (argValue[4], "%s", temp);
    station_speed = (uint8_t) atoi(temp);
    memset(temp, 0, sizeof(temp));
    
    if (!(device_type < MAX_TYPE)) {
        printf("Invalid 'device_type' parameter.\n");
        return 0;
    }
    if (station_number > 1) {
        printf("Invalid 'station_number' parameter.\n");
        return 0; 
    }
    if (station_speed > 3) {
        printf("Invalid 'station_speed' parameter.\n");
        return 0; 
    }

    for (i = 0; i < nb_detected_devices; i++) {

        if ((devices_list[i].device_type == device_type) && 
            (devices_list[i].device_index == device_index)) {

                if (devices_list[i].fip_speed != station_speed) {

                    if (!((devices_list[i].fip_speed == SPEED_2M5) && 
                        (station_speed == SPEED_5M))) {
                        // Support 2.5 Mbps device with 5 Mbps configuration

                        // else exit
                        printf("Station speed must match with device speed.\n");
                        return 0;
                    }
                }
                is_matched = true;
        }
    }

    if (is_matched) {

        if (adam_connect() == -1) {
            printf("Couldn't open ADAM-5000\n");
            return 0;
        }

        if (station_number == 0)
            station0_process(device_type, device_index, sta_xml_path[station_number][station_speed]);
        else
            station1_process(device_type, device_index, sta_xml_path[station_number][station_speed]);

        adam_disconnect();

    } else
        printf("No device was found with device_type (%d) and device_index (%d).\n", device_type, device_index);

    return 0;
}

void station0_process(uint8_t device_type, uint8_t device_index, char* xml_path)
{
    uint32_t ret = SUCCESS;
    USERDEF_TAG tag_conf = {0};
    FIP_USERCONF user_conf = {0};
    PRODUCED_VARIABLE p_var = {0};
    unsigned long tstart_1s, tstart_10s, tcurrent;
    fipcore_device_handle* sta0_hnd = NULL;

    // OUT Tags
    float    TAPumpTemperature = 0;
    // IN Tags
    float    TBPumpTemperature = 0;

    printf("/*****************************************************************************/\n");
    printf("/*                          STATION 0 TEST                                   */\n");
    printf("/*****************************************************************************/\n");

    sta0_hnd = fipcore_device_open(device_type, device_index, &ret);
    if (ret != SUCCESS) {
        printf("STATION 0 - %s\n", fipcore_get_error_str(ret));
        goto _exit1;
    }

    ret = fipcore_init_network(sta0_hnd, XML, xml_path, NULL);
    if (ret != SUCCESS)
        goto _exit;

    // Optional Configuration Information - 
    // (To execute after fipcore_init_network function)
    //fipcore_get_config(sta0_hnd, &user_conf);
    //print_conf(&user_conf);

    ret = fipcore_start_network(sta0_hnd);
    if (ret != SUCCESS)
        goto _exit;

    ret = fipcore_start_ba(sta0_hnd, 1);
    if (ret != SUCCESS)
        goto _exit;

    // TB.PumpTemperature
    ret = fipcore_get_tag_config_by_name(sta0_hnd, "TB.PumpTemperature", &tag_conf);
    if (ret != SUCCESS)
        goto _exit;
    print_tag_conf(&tag_conf);

    // Output variables init
    fipcore_write_var_by_id(sta0_hnd, 0x0001, &p_var);
    fipcore_write_var_by_id(sta0_hnd, 0x8426, &p_var);
    fipcore_write_var_by_id(sta0_hnd, 0x8322, &p_var);
    
    tstart_1s  = GetTickCount();
    tstart_10s = GetTickCount();

    while (1) {
        // MAIN LOOP - EACH 5 MILLISECONDS

        // Get current tick
        tcurrent = GetTickCount();

        if ((tcurrent - tstart_10s) > 10000) {    // Each 10s

            printf("SECTION LOOP - EACH 10 SECONDS\n");

            // Device Medium is read
            medium_processing(sta0_hnd);
            printf("\n");

            tstart_10s = GetTickCount();
        }

        if ((tcurrent - tstart_1s) > 1000) {     // Each 1s

            printf("SECTION LOOP - EACH 1 SECOND\n");

            // OUT Tags - ID (0x8426) - Tags related to Tank A
            TAPumpTemperature = getTemperatureFloat();
            ret = fipcore_write_tag_by_name(sta0_hnd, "TA.PumpTemperature", (void*) &TAPumpTemperature);
            if (ret != SUCCESS)
                printf("fipcore_write_tag_by_name - TA.PumpTemperature - %s\n", fipcore_get_error_str(ret));
            else
                printf("  OUT ID(0x8426) - TA.PumpTemperature : %f Celsius\n", TAPumpTemperature);

            // IN Tags - ID (0x8427) - Tags related to Tank B
            TBPumpTemperature = 0;
            ret = fipcore_read_tag_by_name(sta0_hnd, "TB.PumpTemperature", (void*) &TBPumpTemperature);
            if (ret != SUCCESS)
                printf("fipcore_read_tag_by_name - TB.PumpTemperature - %s\n", fipcore_get_error_str(ret));
            else
                printf("  IN  ID(0x8427) - TB.PumpTemperature : %f Celsius\n", TBPumpTemperature);

            tstart_1s = GetTickCount();

            printf("\n");
        }

        event_processing(sta0_hnd); 
        
        if (kbhit())
            break;
        
#if defined(_WIN32) || defined(_WIN64)
        Sleep(5);
#else
        usleep(5000);
#endif
    }
    
    ret = fipcore_stop_ba(sta0_hnd);

_exit:  
    if (ret != SUCCESS)
        printf("STATION 0 - %s\n", fipcore_get_error_str(ret));

    ret = fipcore_stop_network(sta0_hnd);
    if (ret != SUCCESS)
        printf("STATION 0 - %s\n", fipcore_get_error_str(ret));
_exit1:
    ret = fipcore_device_close(sta0_hnd);
    if (ret != SUCCESS)
        printf("STATION 0 - %s\n", fipcore_get_error_str(ret));
}

void station1_process(uint8_t device_type, uint8_t device_index, char* xml_path)
{
    uint32_t ret = SUCCESS;
    FIP_USERCONF user_conf = {0};
    PRODUCED_VARIABLE p_var = {0};
    unsigned long tstart_1s, tstart_10s, tcurrent;
    fipcore_device_handle* sta1_hnd = NULL;

    // OUT Tags
    float    TBPumpTemperature = 0;
    // IN Tags
    float    TAPumpTemperature = 0;

    printf("/*****************************************************************************/\n");
    printf("/*                          STATION 1 TEST                                   */\n");
    printf("/*****************************************************************************/\n");

    sta1_hnd = fipcore_device_open(device_type, device_index, &ret);
    if (ret != SUCCESS) {
        printf("STATION 1 - %s\n", fipcore_get_error_str(ret));
        goto _exit1;
    }

    ret = fipcore_init_network(sta1_hnd, XML, xml_path, NULL);
    if (ret != SUCCESS)
        goto _exit;

    // Optional Configuration Information - 
    // (To execute after fipcore_init_network function)
    //fipcore_get_config(sta1_hnd, &user_conf);
    //print_conf(&user_conf);

    ret = fipcore_start_network(sta1_hnd);
    if (ret != SUCCESS)
        goto _exit;

    ret = fipcore_start_ba(sta1_hnd, 1);
    if (ret != SUCCESS)
        goto _exit;

    // Output variables init
    fipcore_write_var_by_id(sta1_hnd, 0x8427, &p_var);

    while (1) {
        // MAIN LOOP - EACH 5 MILLISECONDS

        if ((tcurrent - tstart_10s) > 10000) {    // Each 10s

            printf("SECTION LOOP - EACH 10 SECONDS\n");

            // Device Medium is read
            medium_processing(sta1_hnd);
            printf("\n");
            tstart_10s = GetTickCount();
        }

        if ((tcurrent - tstart_1s) > 1000) {     // Each 1s

            printf("SECTION LOOP - EACH 1 SECOND\n");

            // OUT Tags - ID (0x8427) - Tags related to Tank B
            TBPumpTemperature = getTemperatureFloat();
            ret = fipcore_write_tag_by_name(sta1_hnd, "TB.PumpTemperature", (void*) &TBPumpTemperature);
            if (ret != SUCCESS)
                printf("fipcore_write_tag_by_name - TB.PumpTemperature - %s\n", fipcore_get_error_str(ret));
            else
                printf("  OUT ID(0x8427) - TB.PumpTemperature : %f Celsius\n", TBPumpTemperature);

            // IN Tags - ID (0x8426) - Tags related to Tank A
            TAPumpTemperature = 0;
            ret = fipcore_read_tag_by_name(sta1_hnd, "TA.PumpTemperature", (void*) &TAPumpTemperature);
            if (ret != SUCCESS)
                printf("fipcore_read_tag_by_name - TA.PumpTemperature - %s\n", fipcore_get_error_str(ret));
            else
                printf("  IN  ID(0x8426) - TA.PumpTemperature : %f Celsius\n", TAPumpTemperature);

            tstart_1s = GetTickCount();

            printf("\n");
        }

        event_processing(sta1_hnd); 
        
        if (kbhit())
            break;
        
#if defined(_WIN32) || defined(_WIN64)
        Sleep(5);
#else
        usleep(5000);
#endif
    }
    
    ret = fipcore_stop_ba(sta1_hnd);

_exit:  
    if (ret != SUCCESS)
        printf("STATION 1 - %s\n", fipcore_get_error_str(ret));

    ret = fipcore_stop_network(sta1_hnd);
    if (ret != SUCCESS)
        printf("STATION 1 - %s\n", fipcore_get_error_str(ret));
_exit1:
    ret = fipcore_device_close(sta1_hnd);
    if (ret != SUCCESS)
        printf("STATION 1 - %s\n", fipcore_get_error_str(ret));
}

void event_processing(fipcore_device_handle* hnd)
{
    uint32_t ret = SUCCESS;
    uint16_t event_code = 0;
    
    ret = fipcore_read_evt(hnd, &event_code, &event_parameter);

    while (ret == SUCCESS) { // All events are popped

        bool msgReceivedToTreat = false;

        printf("  fipcore_read_evt - ");
        switch (event_code) {
        case FIP_EVT_SEND_VAR_P:    printf("SEND_VAR_P - AccessKey: 0x%04x\n", event_parameter); break;
        case FIP_EVT_SEND_VAR_T:    printf("SEND_VAR_T - AccessKey: 0x%04x\n", event_parameter); break;
        case FIP_EVT_RECEIVE_VAR_P: printf("RECEIVE_VAR_P\n"); break;
        case FIP_EVT_RECEIVE_VAR_T: printf("RECEIVE_VAR_T\n"); break;
        case FIP_EVT_RECEIVE_MSG:   printf("RECEIVE_MSG\n"); msgReceivedToTreat = true; break;
        case FIP_EVT_SEND_MSG:      printf("SEND_MSG - Report: 0x%04x\n", event_parameter); break;
        case FIP_EVT_SEND_APU:      printf("SEND_APU\n"); break;
        case FIP_EVT_SEND_APN:      printf("SEND_APN\n"); break;
        case FIP_EVT_BA_ACTIVITY:   printf("BA_ACTIVITY\n"); break;
        case FIP_EVT_BA_STOP1:      printf("BA_STOP1\n"); break;
        case FIP_EVT_BA_STOP2:      printf("BA_STOP2\n"); break;
        case FIP_EVT_BA_STOP3:      printf("BA_STOP3\n"); break;
        case FIP_EVT_BA_IDLE:       printf("BA_IDLE\n"); break;
        default:
            if ((event_code & 0x0500) == 0x0500) {
                uint8_t suspend_number = (uint8_t) (event_code & 0x00FF);
                printf("BA_SUSPEND%d - BA Address Program: 0x%04x\n", suspend_number, event_parameter);
            } else
                printf("UNKNOWN - EventCode: 0x%04x - EventParameter: 0x%04x\n", event_code, event_parameter);          

        break;
        }

        if (msgReceivedToTreat) {
            uint32_t ret1 = SUCCESS;
            uint16_t i = 0;
            MESSAGE_FRAME_IN c_msg = {0};

            ret = fipcore_read_msg(hnd, &c_msg); 
            if (ret == SUCCESS) {

                uint8_t srcSeg  = c_msg.Src_Segment;
                uint8_t destSeg = c_msg.Dest_Segment;
                uint16_t srcId  = (((uint16_t) c_msg.Src_Add_PF) << 8)  + c_msg.Src_Add_Pf;
                uint16_t destId = (((uint16_t) c_msg.Dest_Add_PF) << 8) + c_msg.Dest_Add_Pf;

                printf("  fipcore_read_msg - From ID(0x%04x) to ID(0x%04x) - ", srcId, destId);

                printf("MSG: ");
                for (i = 0; i < c_msg.Msg_Length; i++)
                    printf("%c", c_msg.Useful_Data[i]);
                printf("\n");
            } else
                printf("ERROR - %s\n", fipcore_get_error_str(ret));
        }

        ret = fipcore_read_evt(hnd, &event_code, &event_parameter);
    }
}

void medium_processing(fipcore_device_handle* hnd)
{
    uint32_t ret = SUCCESS;
    FIP_NETWORK_STATUS  fipNetworkStatus = {0};

    // Lecture du medium
    ret = fipcore_get_medium(hnd, &fipNetworkStatus);
    if (ret != SUCCESS)
        printf("  fipcore_get_medium error: %08x", ret);
    else {
        bool ch1Active = ((fipNetworkStatus.FIELDUAL_Status & MEDIUM_STATUS_CH1_VALID) == MEDIUM_STATUS_CH1_VALID);
        bool ch2Active = ((fipNetworkStatus.FIELDUAL_Status & MEDIUM_STATUS_CH2_VALID) == MEDIUM_STATUS_CH2_VALID);
        bool ch1TXError = ((fipNetworkStatus.FIELDUAL_Status & MEDIUM_STATUS_CH1_TX_ERROR) == MEDIUM_STATUS_CH1_TX_ERROR);
        bool ch2TXError = ((fipNetworkStatus.FIELDUAL_Status & MEDIUM_STATUS_CH2_TX_ERROR) == MEDIUM_STATUS_CH2_TX_ERROR);
        bool ch1WatchdogError = ((fipNetworkStatus.FIELDUAL_Status & MEDIUM_STATUS_CH1_WATCHDOG) == MEDIUM_STATUS_CH1_WATCHDOG);
        bool ch2WatchdogError = ((fipNetworkStatus.FIELDUAL_Status & MEDIUM_STATUS_CH2_WATCHDOG) == MEDIUM_STATUS_CH2_WATCHDOG);
        char baStatusStr[255];

        switch (fipNetworkStatus.BA_Status) {
            case BA_STATUS_STOPPED:        strcpy(baStatusStr, "STOPPED"); break;
            case BA_STATUS_STARTING:       strcpy(baStatusStr, "STARTING"); break;
            case BA_STATUS_IDLE:           strcpy(baStatusStr, "IDLE"); break;
            case BA_STATUS_MACRO_END:      strcpy(baStatusStr, "MACRO END"); break;
            case BA_STATUS_SENDING_ID_DAT: strcpy(baStatusStr, "SENDING ID_DAT"); break;
            case BA_STATUS_SENDING_ID_MSG: strcpy(baStatusStr, "SENDING ID_MSG"); break;
            case BA_STATUS_TESTING:        strcpy(baStatusStr, "TESTING"); break;
            case BA_STATUS_TESTING_END:    strcpy(baStatusStr, "TESTING END"); break;
            case BA_STATUS_PENDING:        strcpy(baStatusStr, "PENDING"); break;
            case BA_STATUS_APER_WINDOW:    strcpy(baStatusStr, "APER. VAR WINDOW"); break;
            case BA_STATUS_MSG_WINDOW:     strcpy(baStatusStr, "APER. MSG WINDOW"); break;
            case BA_STATUS_WAITING_TIME:   strcpy(baStatusStr, "WAITING TIME"); break;
            case BA_STATUS_WAITING_SYNC:   strcpy(baStatusStr, "WAITING SYNC."); break;
            case BA_STATUS_WAITING_SYNC_SILENT: strcpy(baStatusStr, "WAITING SYNC. SILENT"); break;
            default: strcpy(baStatusStr, "UNKNOWN"); break;

        }

        printf("  fipcore_get_medium\n");
        printf("    No_Error_Channel1   : %d\n", fipNetworkStatus.No_Error_Channel1);
        printf("    No_Error_Channel2   : %d\n", fipNetworkStatus.No_Error_Channel2);
        printf("    Error_Channel1      : %d\n", fipNetworkStatus.Error_Channel1);
        printf("    Error_Channel2      : %d\n", fipNetworkStatus.Error_Channel2);
        printf("    Error_No_Echo       : %d\n", fipNetworkStatus.Error_No_Echo);
        printf("    Error_Testp_Channel1: %d\n", fipNetworkStatus.Error_Testp_Channel1);
        printf("    Error_Testp_Channel2: %d\n", fipNetworkStatus.Error_Testp_Channel2);
        printf("    BA_Current_Macrocyle: %04xh\n", fipNetworkStatus.BA_Current_Macrocyle);
        printf("    BA_Status           : %04xh - [%s]\n", fipNetworkStatus.BA_Status, baStatusStr);
        printf("    FIELDUAL_Status     : %04xh\n", fipNetworkStatus.FIELDUAL_Status);
        printf("          [Channel 1 - Active: %d - TX Error: %d - Watchdog Error: %d]\n", ch1Active, ch1TXError, ch1WatchdogError);
        printf("          [Channel 2 - Active: %d - TX Error: %d - Watchdog Error: %d]\n", ch2Active, ch2TXError, ch2WatchdogError);


    }
}

void print_device_info(FIP_DEVICECONF* device_conf)
{
    switch(device_conf->device_type) {
    case PCI_PCIE_TYPE:     printf("   device_type     : PCI_PCIE_TYPE \n"); break;
    case USB_TYPE:          printf("   device_type     : USB_TYPE \n"); break;
    default:                printf("   device_type     : UNKNOWN_TYPE \n"); break;
    }
    printf("   device_index    : %d \n", device_conf->device_index);
    printf("   product_number  : 0x%08x \n", device_conf->product_number);
    printf("   device_version  : %d.%d.%d \n", device_conf->device_version.major,
                                            device_conf->device_version.minor,
                                            device_conf->device_version.revision);
    switch(device_conf->fip_speed) {
    case SPEED_31K25:   printf("   fip_speed       : SPEED_31K25 \n"); break;
    case SPEED_1M:      printf("   fip_speed       : SPEED_1M \n"); break;
    case SPEED_2M5:     printf("   fip_speed       : SPEED_2M5 or SPEED_5M \n"); break;
    default:            printf("   fip_speed       : UNKNOWN_SPEED \n"); break;
    }
    switch(device_conf->fip_impedance) {
    case IMPEDANCE_150: printf("   fip_impedance   : IMPEDANCE_150 \n\n"); break;
    case IMPEDANCE_120: printf("   fip_impedance   : IMPEDANCE_120 \n\n"); break;
    default:            printf("   fip_impedance   : UNKNOWN_IMPEDANCE \n\n"); break;
    }
}

void print_tag_conf(const USERDEF_TAG* ut) {
    if (ut != NULL) {

        printf("   tag_name           : %s \n", ut->tag_name);
        printf("   tag_id             : %d \n", ut->tag_id);
        printf("   type_name          : %s \n", ut->type_name);
        printf("   type_id            : %d \n", ut->type_id);
        printf("   access_rights      : %d \n", ut->access_rights);
        printf("   description        : %s \n", ut->description);
        printf("   identifier         : 0x%04x \n", ut->identifier);
        printf("   start_bit          : %d \n", ut->start_bit);
        printf("   stop_bit           : %d \n", ut->stop_bit);
        printf("   endianness         : %d \n", ut->endianness);
    } else {
        printf("    - user_def_tag    : \n");
        printf("        - unallocated \n");
    }
}

void print_msg_conf(const USERDEF_MESSAGE* um) {
    if (um != NULL) {
        printf("   type               : %d \n", um->type);
        printf("   channel            : %d \n", um->channel);
        printf("   acknowledgment     : %d \n", um->acknowledgment);
        printf("   destination_identifier : 0x%04x \n", um->destination_identifier);
        printf("   destination_segment: 0x%02x \n", um->destination_segment);
        printf("   source_identifier  : 0x%04x \n", um->source_identifier);
        printf("   source_segment     : 0x%02x \n", um->source_segment);
    } else {
        printf("    - user_def_msg    : \n");
        printf("        - unallocated \n");
    }
}

void print_conf(const FIP_USERCONF* fuc)
{
    uint16_t cp = 0;
    uint16_t i  = 0;

    printf("/*****************************************************************************/\n");
    printf("/*                              FIP_USERCONF                                 */\n");
    printf("/*****************************************************************************/\n");
    if (fuc != NULL) {
        printf("    - this_station               : \n");
        printf("        - subscriber_number      : %d \n", fuc->this_station->subscriber_number);
        printf("        - subscriber_name        : %s \n", fuc->this_station->subscriber_name);
        printf("    - speed                      : %d \n", fuc->speed);
        printf("    - t_slot                     : %d \n", fuc->t_slot);
        printf("    - silence                    : %d \n", fuc->silence);
        printf("    - turn_around                : %d \n", fuc->turn_around);
        printf("    - frame_type                 : %d \n", fuc->frame_type);
        printf("    - is_arbiter                 : %d \n", fuc->is_arbiter);
        printf("    - arbiter                    : \n");
        printf("        - ba_max_subscriber_number : %d \n", fuc->arbiter->ba_max_subscriber_number);
        printf("        - ba_priority            : %d \n", fuc->arbiter->ba_priority);
        printf("        - ba_max_priority        : %d \n", fuc->arbiter->ba_max_priority);
        printf("        - ba_optimization        : %d \n", fuc->arbiter->ba_optimization);
        printf("        - nb_ba_1_instructions   : %d \n", cp = fuc->arbiter->nb_ba_1_instructions);
        printf("        - ba_1_instructions      : \n");
        {
            INSTRUCTION* instructions = fuc->arbiter->ba_1_instructions;
            for (i = 0; i < cp; i++)
                printf("            - %s \n", (instructions + i)->op_code);
        }
        printf("        - nb_ba_2_instructions   : %d \n", cp = fuc->arbiter->nb_ba_2_instructions);
        printf("        - ba_2_instructions      : \n");
        {
            INSTRUCTION* instructions = fuc->arbiter->ba_2_instructions;
            for (i = 0; i < cp; i++)
                printf("            - %s \n", (instructions + i)->op_code);
        }   
        printf("    - nb_variables               : %d \n", cp = fuc->nb_variables);
        printf("    - variables                  : \n");
        {
            for (i = 0; i < cp; i++) {
                printf("        - number %d \n", i);
                printf("            - name               : %s \n", fuc->variables[i].name);
                printf("            - identifier         : 0x%04x \n", fuc->variables[i].identifier);
                printf("            - pdu                : 0x%04x \n", fuc->variables[i].pdu);
                printf("            - length             : %d \n", fuc->variables[i].length);
                printf("            - type               : %d \n", fuc->variables[i].type);
                printf("            - refreshment_type   : %d \n", fuc->variables[i].refreshment_type);
                printf("            - refreshment_period : 0x%04x \n", fuc->variables[i].refreshment_period);
                printf("            - promptitude_type   : %d \n", fuc->variables[i].promptitude_type);
                printf("            - promptitude_period : 0x%04x \n", fuc->variables[i].promptitude_period);
                printf("            - msg_channel        : %d \n", fuc->variables[i].msg_channel);
                printf("            - messages_service   : 0x%02x \n", fuc->variables[i].messages_service);
            }
        }
        printf("    - nb_messages                : %d \n", cp = fuc->nb_messages);
        printf("    - messages                   : \n");
        {
            for (i = 0; i < cp; i++) {
                printf("        - number %d \n", i);
                printf("            - name               : %s \n", fuc->messages[i].name);
                printf("            - type               : %d \n", fuc->messages[i].type);
                printf("            - channel            : %d \n", fuc->messages[i].channel);
                printf("            - acknowledgment     : %d \n", fuc->messages[i].acknowledgment);
                printf("            - destination_identifier: 0x%04x  \n", fuc->messages[i].destination_identifier);
                printf("            - destination_segment: 0x%02x  \n", fuc->messages[i].destination_segment);
                printf("            - source_identifier  : 0x%04x  \n", fuc->messages[i].source_identifier);
                printf("            - source_segment     : 0x%02x  \n", fuc->messages[i].source_segment);
            }
        }
        printf("     - nb_tags            : %d \n", cp = fuc->nb_tags);
        printf("     - tags               : \n");
        {
            for (i = 0; i < cp; i++) {
                printf("        - number %d \n", i);
                printf("            - tag_name      : %s \n", fuc->tags[i].tag_name);
                printf("            - tag_id        : %d \n", fuc->tags[i].tag_id);
                printf("            - type_name     : %s \n", fuc->tags[i].type_name);
                printf("            - type_id       : %d \n", fuc->tags[i].type_id);
                printf("            - access_rights : %d \n", fuc->tags[i].access_rights);
                printf("            - description   : %s \n", fuc->tags[i].description);
                printf("            - identifier    : 0x%04x \n", fuc->tags[i].identifier);
                printf("            - start_bit     : %d \n", fuc->tags[i].start_bit);
                printf("            - stop_bit      : %d \n", fuc->tags[i].stop_bit);
                printf("            - endianness    : %d \n", fuc->tags[i].endianness);
            }
        }
    } else {
        printf("    - user_conf               : \n");
        printf("        - unallocated \n");
    }
}