//  Created on : 13 Feb 2017
//  Author : M. Coroyer
//  Company : © 2017 Exoligent
//  Description:
//      This is an example of using the FipCore library.
//      The xml stations files are built from fiplabs labpro files:
//          - '2sta_worldfip_31K25.labpro'  : 31.25 Kbps
//              -> sta0_worldfip_31K25.xml
//              -> sta1_worldfip_31K25.xml
//          - '2sta_worldfip_1M.labpro'     : 1     Mbps
//              -> sta0_worldfip_1M.xml
//              -> sta1_worldfip_1M.xml
//          - '2sta_worldfip_2M5.labpro'    : 2.5   Mbps
//              -> sta0_worldfip_2M5.xml
//              -> sta1_worldfip_2M5.xml
//          - '2sta_worldfip_5M.labpro'     : 5     Mbps
//              -> sta0_worldfip_5M.xml
//              -> sta1_worldfip_5M.xml
//
//  Station0 (sta0) Configuration
//      - Arbiter: true - 1 Scan Table
//          -> Scan Table 1:
//                  SEND_ID_DAT (0x8426)
//                  SEND_ID_DAT (0x8427)
//                  SEND_ID_DAT (0x0001)
//                  SEND_MSG (5 ms)
//                  SEND_APER (10 ms)
//                  WAIT_TIME (15 ms)
//                  NEXT_MACRO ()
//      - Periodic IDs
//          -> ID(0x0001) :
//                  Type:                               Out
//                  Length:                             120 bytes
//                  Refreshment:                        true (200ms)
//                  Aperiodic's List Transfert Request: true
//                  Message Emission Channel:           false
//                  Message Reception Channel:          true
//          -> ID(0x8426) :
//                  Type:                               Out
//                  Length:                             12 bytes
//                  Refreshment:                        true (50ms)
//                  Aperiodic's List Transfert Request: false
//                  Message Emission Channel:           true (Type: Aperiodic - Channel: 0)
//                  Message Reception Channel:          false
//          -> ID(0x8427) :
//                  Type:                               In
//                  Promptness:                         true (50ms)
//                  Message Reception Channel:          false
//                  Other details: see Station1 (sta1) - ID(0x8427) 
//      - Aperiodic IDs
//          -> ID(0x8322) :
//                  Type:                               Out
//                  Length:                             10 bytes
//                  Refreshment:                        false
//                  Aperiodic's List Transfert Request: false
//                  Message Emission Channel:           false
//                  Message Reception Channel:          false
//      - Message(s)
//          -> ID (0x8426) to ID (0x8427)
//                  Source Address:                     Segment 0 - ID 0x8426
//                  Destination Address:                Segment 0 - ID 0x8427
//                  Acknowledgment:                     true
//                  Channel Emission Attached:          Type: Aperiodic - Channel: 0 (see ID(0x8426))
//
//  Station1 (sta1) Configuration
//      - Arbiter: true - 1 Scan Table
//          -> Scan Table 1:
//                  SEND_ID_DAT (0x8426)
//                  SEND_ID_DAT (0x8427)
//                  SEND_ID_DAT (0x0001)
//                  SEND_MSG (5 ms)
//                  WAIT_TIME (5 ms)
//                  NEXT_MACRO ()
//      - Periodic IDs
//          -> ID(0x8426) :
//                  Type:                               In
//                  Promptness:                         true (50ms)
//                  Message Reception Channel:          false
//                  Other details: see Station0 (sta0) - ID(0x8426) 
//          -> ID(0x8427) :
//                  Type:                               Out
//                  Length:                             12 bytes
//                  Refreshment:                        true (200 ms)
//                  Aperiodic's List Transfert Request: false
//                  Message Emission Channel:           false
//                  Message Reception Channel:          true
//      - Aperiodic IDs
//          -> ID(0x8322) :
//                  Type:                               In
//                  Promptness:                         true (50ms)
//                  Message Reception Channel:          false
//                  Other details: see Station0 (sta0) - ID(0x8322)
//
//  In this example:
//      station0_process routine:
//          Open Device / Load Configuration / Run Bus Arbiter
//          Loop
//              Each 5ms
//                  Events are popped (message and aperiodic var reception are treat here)
//                  OPC UA Server iterate (execute an iteration of the server's main loop)
//              Each 1s 
//                  out tag: 
//                      Randoms values for the following tags:
//                      "TA.PumpTemperature", "TA.PumpAmps", "TA.PumpPressure"
//                       "TA.PumpAlarm", "TA.Volume".
//                      Note: These tags are defined in OUT ID (0x8426)
//                  in tag: 
//                      Tags values from ID (0x8427) are read:
//                      "TB.PumpTemperature", "TB.PumpAmps", "TB.PumpPressure"
//                       "TB.PumpAlarm", "TB.Volume".
//              Each 10s 
//                  "TA.Notification" value tag is incremented (10 bytes ascii in ID (0x8322))
//                  Aperiodic Request (ID (0x8322)) is sent
//                  Device Medium is read
//          End Loop on key press
//
//      station1_process routine:
//          Open Device / Load Configuration / Run Bus Arbiter
//          Loop
//              Each 5ms
//                  Events are popped (message and aperiodic var reception are treat here)
//                  OPC UA Server iterate (execute an iteration of the server's main loop)
//              Each 1s
//                  out tag: 
//                      Randoms values for the following tags:
//                      "TB.PumpTemperature", "TB.PumpAmps", "TB.PumpPressure"
//                       "TB.PumpAlarm", "TB.Volume".
//                      Note: These tags are defined in OUT ID (0x8427)
//                  in tag: 
//                      Tags values from ID (0x8426) and ID (0x8322) are read:
//                      "TA.PumpTemperature", "TA.PumpAmps", "TA.PumpPressure"
//                       "TA.PumpAlarm", "TA.Volume", "TA.Notification"
//              Each 10s 
//                  Device Medium is read
//          End Loop on key press
//
//  Note: In the example, ID(0x0001) is only used to treat the aperiodic's list requests; that's why it is
//  present in periodic window in scan table


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "fipcore.h"

#if defined(_WIN32) || defined(_WIN64)
#   include <windows.h>
#   include <conio.h>
#elif defined(__linux__) || defined(__APPLE__) || defined(__QNX__)
#   include <unistd.h>     // usleep()
#   include <termios.h>
#   include <sys/types.h>
#   include <sys/time.h>

void changemode(int dir)
{
    static struct termios oldt, newt;
    
    if (dir == 1) {
        tcgetattr(STDIN_FILENO, &oldt);
        newt = oldt;
        newt.c_lflag &= ~(ICANON | ECHO);
        tcsetattr(STDIN_FILENO, TCSANOW, &newt);
    } else
        tcsetattr(STDIN_FILENO, TCSANOW, &oldt);
}

int kbhit(void)
{
    struct timeval tv;
    fd_set rdfs;
    
    tv.tv_sec = 0;
    tv.tv_usec = 0;
    
    FD_ZERO(&rdfs);
    FD_SET(STDIN_FILENO, &rdfs);
    
    select(STDIN_FILENO + 1, &rdfs, NULL, NULL, &tv);
    return FD_ISSET(STDIN_FILENO, &rdfs);
}
#endif


uint16_t event_parameter = 0;

char* sta_xml_path[2][4] = {
    {"../xml/sta0_worldfip_31K25.xml", "../xml/sta0_worldfip_1M.xml", "../xml/sta0_worldfip_2M5.xml", "../xml/sta0_worldfip_5M.xml"},
    {"../xml/sta1_worldfip_31K25.xml", "../xml/sta1_worldfip_1M.xml", "../xml/sta1_worldfip_2M5.xml", "../xml/sta1_worldfip_5M.xml"}
};

void station0_process(uint8_t device_type, uint8_t device_index, char* xml_path);
void station1_process(uint8_t device_type, uint8_t device_index, char* xml_path);
void print_device_info(FIP_DEVICECONF* device_conf);
void print_conf(const FIP_USERCONF* fuc);
void event_processing(fipcore_device_handle* hnd);
void medium_processing(fipcore_device_handle* hnd);

float randomTemperatureFloat()
{
    // 90 °C +/- 10 °C
    int signT = ((rand() % 2) == 0) ? 1. : -1.;
    float r = 90. + signT * ((float)rand()/(float)RAND_MAX) * 10.;
    return r;
}

float randomAmpsFloat()
{
    // 0 - 70 A
    float r = rand() % 70; + ((float)rand()/(float)RAND_MAX);
    return r;
}

uint16_t randomPressureUInt16()
{
    // 80 psi +/- 40 psi
    int signP = ((rand() % 2) == 0) ? 1 : -1;
    uint16_t r = 80 + signP * ((uint16_t)((float)rand()* 100.) % 40);
    return r;
}

int main(int argCount, char *argValue[])
{
    uint32_t ret = SUCCESS;
    uint8_t i = 0;
    uint8_t device_type;
    uint8_t device_index;
    uint8_t station_number;
    uint8_t station_speed;
    FIP_DEVICECONF devices_list[20];
    uint8_t nb_detected_devices = 0;
    bool    is_matched = false;
    char    xml_path[255];
    char    temp[10];

    // Device Enumeration
    printf("/*****************************************************************************/\n");
    printf("/*                          DEVICES ON COMPUTER                              */\n");
    printf("/*****************************************************************************/\n");

    ret = fipcore_get_devices_number(&nb_detected_devices);
    if (ret != SUCCESS) {
        printf("fipcore_get_devices_number - %s\n", fipcore_get_error_str(ret));
        return 0;
    }

    printf("NUMBER OF DETECTED DEVICES - %d\n", nb_detected_devices);

    ret = fipcore_get_devices_list(devices_list, &nb_detected_devices);
    if (ret != SUCCESS) {
        printf("fipcore_get_devices_list - %s\n", fipcore_get_error_str(ret));
        return 0;
    }

    for (i = 0; i < nb_detected_devices; i++)
        print_device_info(&(devices_list[i]));

    if (argCount < 4) {
        printf("This application must have this form: fipcore_opcua_nonblocking.exe 'device_type' 'device_index' 'station_number' 'station_speed' \n");
        printf("'device_type': 1: PCI_PCIE_TYPE - 2: USB_TYPE\n");
        printf("'device_index': 0 to 255\n");
        printf("'station_number': 0: Station 0 - 1: Station 1\n");
        printf("'station_speed': 0: 31.25 Kbps - 1: 1 Mbps - 2: 2.5 Mbps - 3: 5Mbps\n");
        return 0;
    }

    sscanf (argValue[1], "%s", temp);
    device_type = (uint8_t) atoi(temp);
    memset(temp, 0, sizeof(temp));

    sscanf (argValue[2], "%s", temp);
    device_index = (uint8_t) atoi(temp);
    memset(temp, 0, sizeof(temp));

    sscanf (argValue[3], "%s", temp);
    station_number = (uint8_t) atoi(temp);
    memset(temp, 0, sizeof(temp));

    sscanf (argValue[4], "%s", temp);
    station_speed = (uint8_t) atoi(temp);
    memset(temp, 0, sizeof(temp));
    
    if (!(device_type < MAX_TYPE)) {
        printf("Invalid 'device_type' parameter.\n");
        return 0;
    }
    if (station_number > 1) {
        printf("Invalid 'station_number' parameter.\n");
        return 0; 
    }
    if (station_speed > 3) {
        printf("Invalid 'station_speed' parameter.\n");
        return 0; 
    }

    for (i = 0; i < nb_detected_devices; i++) {

        if ((devices_list[i].device_type == device_type) && 
            (devices_list[i].device_index == device_index)) {

                if (devices_list[i].fip_speed != station_speed) {

                    if (!((devices_list[i].fip_speed == SPEED_2M5) && 
                        (station_speed == SPEED_5M))) {
                        // Support 2.5 Mbps device with 5 Mbps configuration

                        // else exit
                        printf("Station speed must match with device speed.\n");
                        return 0;
                    }
                }
                is_matched = true;
        }
    }

    if (is_matched) {

        if (station_number == 0)
            station0_process(device_type, device_index, sta_xml_path[station_number][station_speed]);
        else
            station1_process(device_type, device_index, sta_xml_path[station_number][station_speed]);
    } else
        printf("No device was found with device_type (%d) and device_index (%d).\n", device_type, device_index);

    return 0;
}

void station0_process(uint8_t device_type, uint8_t device_index, char* xml_path)
{
    uint32_t ret = SUCCESS;
    FIP_USERCONF user_conf = {0};
    PRODUCED_VARIABLE p_var = {0};
    uint16_t loop_counter_1s = 200;     // 5ms multiple
    uint16_t loop_counter_10s = 2000;   // 5ms multiple
    uint16_t opc_timeout = 0;
    time_t   t;
    fipcore_device_handle* sta0_hnd = NULL;

    // OUT Tags
    uint8_t  TAVolume= 50;
    float    TAPumpTemperature = 0;
    bool     TAPumpAlarm  = false;
    float    TAPumpAmps = 0;
    uint16_t TAPumpPressure = 0;
    char     TANotification[10] = {0};
    uint16_t notificationNumber = 0;

    // IN Tags
    uint8_t  TBVolume= 0;
    float    TBPumpTemperature = 0;
    bool     TBPumpAlarm  = false;
    float    TBPumpAmps = 0;
    uint16_t TBPumpPressure = 0;

    /* Intializes random number generator */
    srand((unsigned) time(&t));

    printf("/*****************************************************************************/\n");
    printf("/*                          STATION 0 TEST                                   */\n");
    printf("/*****************************************************************************/\n");

    sta0_hnd = fipcore_device_open(device_type, device_index, &ret);
    if (ret != SUCCESS) {
        printf("STATION 0 - %s\n", fipcore_get_error_str(ret));
        goto _exit1;
    }

    ret = fipcore_init_network(sta0_hnd, XML, xml_path, NULL);
    if (ret != SUCCESS)
        goto _exit;

    // Optional Configuration Information - 
    // (To execute after fipcore_init_network function)
    //fipcore_get_config(sta0_hnd, &user_conf);
    //print_conf(&user_conf);

    ret = fipcore_start_network(sta0_hnd);
    if (ret != SUCCESS)
        goto _exit;

    ret = fipcore_start_ba(sta0_hnd, 1);
    if (ret != SUCCESS)
        goto _exit;

    // Output variables init
    fipcore_write_var_by_id(sta0_hnd, 0x0001, &p_var);
    fipcore_write_var_by_id(sta0_hnd, 0x8426, &p_var);
    fipcore_write_var_by_id(sta0_hnd, 0x8322, &p_var);

#if defined(__linux__) || defined(__APPLE__)
    changemode(1);
#endif
    
    // OPC UA START SERVER
    ret = fipcore_opcua_run_startup(sta0_hnd, 16664);
    if (ret != SUCCESS)
        goto _exit;

    while (1) {
        // MAIN LOOP - EACH 5 MILLISECONDS

        if (loop_counter_10s == 0) {    // Each 10s

            printf("SECTION LOOP - EACH 10 SECONDS\n");

            // Device Medium is read
            medium_processing(sta0_hnd);

            printf("\n");

            // OUT Tag - TA.Notification (Attached to 0x8322 aperiodic variable)
            notificationNumber++;
            sprintf(TANotification, "TANot%d", notificationNumber);
            ret = fipcore_write_tag_by_name(sta0_hnd, "TA.Notification", (void*) TANotification);
            if (ret != SUCCESS)
                printf("fipcore_write_tag_by_name - TA.Notification - %s\n", fipcore_get_error_str(ret));

            // Aperiodic Request (ID (0x8322)) is sent
            ret = fipcore_send_aper_by_id(sta0_hnd, 0x8322, 1); // 1: Urgent Mode
            if (ret != SUCCESS)
                goto _exit;

            printf("  OUT ID(0x8322) - TA.Notification: %s\n", TANotification);

            loop_counter_10s = 2000;
        }

        if (loop_counter_1s == 0) {     // Each 1s

            printf("SECTION LOOP - EACH 1 SECOND\n");

            // OUT Tags - ID (0x8426) - Tags related to Tank A
            TAPumpTemperature = randomTemperatureFloat();
            ret = fipcore_write_tag_by_name(sta0_hnd, "TA.PumpTemperature", (void*) &TAPumpTemperature);
            if (ret != SUCCESS)
                printf("fipcore_write_tag_by_name - TA.PumpTemperature - %s\n", fipcore_get_error_str(ret));
            else
                printf("  OUT ID(0x8426) - TA.PumpTemperature : %f Celsius\n", TAPumpTemperature);

            TAPumpAmps = randomAmpsFloat();
            ret = fipcore_write_tag_by_name(sta0_hnd, "TA.PumpAmps", (void*) &TAPumpAmps);
            if (ret != SUCCESS)
                printf("fipcore_write_tag_by_name - TA.PumpAmps - %s\n", fipcore_get_error_str(ret));
            else
                printf("  OUT ID(0x8426) - TA.PumpAmps        : %f A\n", TAPumpAmps);

            TAPumpPressure = randomPressureUInt16();
            ret = fipcore_write_tag_by_name(sta0_hnd, "TA.PumpPressure", (void*) &TAPumpPressure);
            if (ret != SUCCESS)
                printf("fipcore_write_tag_by_name - TA.PumpPressure - %s\n", fipcore_get_error_str(ret));
            else
                printf("  OUT ID(0x8426) - TA.PumpPressure    : %d psi\n", TAPumpPressure);

            TAPumpAlarm = false;
            if (TAPumpAmps > 60) 
                TAPumpAlarm = true;
            ret = fipcore_write_tag_by_name(sta0_hnd, "TA.PumpAlarm", (void*) &TAPumpAlarm);
            if (ret != SUCCESS)
                printf("fipcore_write_tag_by_name - TA.PumpAlarm - %s\n", fipcore_get_error_str(ret));
            else
                printf("  OUT ID(0x8426) - TA.PumpAlarm       : %d\n", TAPumpAlarm);

            if (TAPumpAmps > 35) {
                TAVolume++;
                if (TAVolume > 100)
                    TAVolume = 100;
            } else {
                if (TAVolume != 0)
                    TAVolume--;
            }
            ret = fipcore_write_tag_by_name(sta0_hnd, "TA.Volume", (void*) &TAVolume);
            if (ret != SUCCESS)
                printf("fipcore_write_tag_by_name - TA.Volume - %s\n", fipcore_get_error_str(ret));
            else
                printf("  OUT ID(0x8426) - TA.Volume          : %d percent\n", TAVolume);

            // IN Tags - ID (0x8427) - Tags related to Tank B
            TBPumpTemperature = 0;
            ret = fipcore_read_tag_by_name(sta0_hnd, "TB.PumpTemperature", (void*) &TBPumpTemperature);
            if (ret != SUCCESS)
                printf("fipcore_read_tag_by_name - TB.PumpTemperature - %s\n", fipcore_get_error_str(ret));
            else
                printf("  IN  ID(0x8427) - TB.PumpTemperature : %f Celsius\n", TBPumpTemperature);

            TBPumpAmps = 0;
            ret = fipcore_read_tag_by_name(sta0_hnd, "TB.PumpAmps", (void*) &TBPumpAmps);
            if (ret != SUCCESS)
                printf("fipcore_read_tag_by_name - TB.PumpAmps - %s\n", fipcore_get_error_str(ret));
            else
                printf("  IN  ID(0x8427) - TB.PumpAmps        : %f A\n", TBPumpAmps);

            TBPumpPressure = 0;
            ret = fipcore_read_tag_by_name(sta0_hnd, "TB.PumpPressure", (void*) &TBPumpPressure);
            if (ret != SUCCESS)
                printf("fipcore_read_tag_by_name - TB.PumpPressure - %s\n", fipcore_get_error_str(ret));
            else
                printf("  IN  ID(0x8427) - TB.PumpPressure    : %d psi\n", TBPumpPressure);

            TBPumpAlarm = false;
            ret = fipcore_read_tag_by_name(sta0_hnd, "TB.PumpAlarm", (void*) &TBPumpAlarm);
            if (ret != SUCCESS)
                printf("fipcore_read_tag_by_name - TB.PumpAlarm - %s\n", fipcore_get_error_str(ret));
            else
                printf("  IN  ID(0x8427) - TB.PumpAlarm       : %d\n", TBPumpAlarm);

            TBVolume = 0;
            ret = fipcore_read_tag_by_name(sta0_hnd, "TB.Volume", (void*) &TBVolume);
            if (ret != SUCCESS)
                printf("fipcore_read_tag_by_name - TB.Volume - %s\n", fipcore_get_error_str(ret));
            else
                printf("  IN  ID(0x8427) - TB.Volume          : %d percent\n", TBVolume);

            loop_counter_1s = 200;

            printf("\n");
        }

        event_processing(sta0_hnd); 
        
        if (kbhit())
            break;
        
#if defined(_WIN32) || defined(_WIN64)
        Sleep(5);
#else
        usleep(5000);
#endif
        loop_counter_10s--;
        loop_counter_1s--;

        ret = fipcore_opcua_run_iterate(sta0_hnd, false, &opc_timeout);
        if (ret != SUCCESS)
            goto _exit;
    }

#if defined(__linux__) || defined(__APPLE__)
    changemode(0);
#endif
    
    ret = fipcore_stop_ba(sta0_hnd);

    ret = fipcore_opcua_run_shutdown(sta0_hnd);

_exit:  
    if (ret != SUCCESS)
        printf("STATION 0 - %s\n", fipcore_get_error_str(ret));

    ret = fipcore_stop_network(sta0_hnd);
    if (ret != SUCCESS)
        printf("STATION 0 - %s\n", fipcore_get_error_str(ret));
_exit1:
    ret = fipcore_device_close(sta0_hnd);
    if (ret != SUCCESS)
        printf("STATION 0 - %s\n", fipcore_get_error_str(ret));
}

void station1_process(uint8_t device_type, uint8_t device_index, char* xml_path)
{
    uint32_t ret = SUCCESS;
    FIP_USERCONF user_conf = {0};
    PRODUCED_VARIABLE p_var = {0};
    uint16_t loop_counter_1s = 200;     // 5ms multiple
    uint16_t loop_counter_10s = 2000;   // 5ms multiple
    uint16_t opc_timeout = 0;
    time_t   t;
    fipcore_device_handle* sta1_hnd = NULL;

    // OUT Tags
    uint8_t  TBVolume= 50;
    float    TBPumpTemperature = 0;
    bool     TBPumpAlarm  = false;
    float    TBPumpAmps = 0;
    uint16_t TBPumpPressure = 0;

    // IN Tags
    uint8_t  TAVolume= 0;
    float    TAPumpTemperature = 0;
    bool     TAPumpAlarm  = false;
    float    TAPumpAmps = 0;
    uint16_t TAPumpPressure = 0;
    char     TANotification[10] = {0};

    /* Intializes random number generator */
    srand((unsigned) time(&t));

    printf("/*****************************************************************************/\n");
    printf("/*                          STATION 1 TEST                                   */\n");
    printf("/*****************************************************************************/\n");

    sta1_hnd = fipcore_device_open(device_type, device_index, &ret);
    if (ret != SUCCESS) {
        printf("STATION 1 - %s\n", fipcore_get_error_str(ret));
        goto _exit1;
    }

    ret = fipcore_init_network(sta1_hnd, XML, xml_path, NULL);
    if (ret != SUCCESS)
        goto _exit;

    // Optional Configuration Information - 
    // (To execute after fipcore_init_network function)
    //fipcore_get_config(sta1_hnd, &user_conf);
    //print_conf(&user_conf);

    ret = fipcore_start_network(sta1_hnd);
    if (ret != SUCCESS)
        goto _exit;

    ret = fipcore_start_ba(sta1_hnd, 1);
    if (ret != SUCCESS)
        goto _exit;

    // Output variables init
    fipcore_write_var_by_id(sta1_hnd, 0x8427, &p_var);

#if defined(__linux__) || defined(__APPLE__)
    changemode(1);
#endif

    // OPC UA START SERVER
    ret = fipcore_opcua_run_startup(sta1_hnd, 16664);
    if (ret != SUCCESS)
        goto _exit;

    while (1) {
        // MAIN LOOP - EACH 5 MILLISECONDS

        if (loop_counter_10s == 0) {    // Each 10s

            printf("SECTION LOOP - EACH 10 SECONDS\n");

            // Device Medium is read
            medium_processing(sta1_hnd);
            printf("\n");
            loop_counter_10s = 2000;
        }

        if (loop_counter_1s == 0) {     // Each 1s

            printf("SECTION LOOP - EACH 1 SECOND\n");

            // OUT Tags - ID (0x8427) - Tags related to Tank B
            TBPumpTemperature = randomTemperatureFloat();
            ret = fipcore_write_tag_by_name(sta1_hnd, "TB.PumpTemperature", (void*) &TBPumpTemperature);
            if (ret != SUCCESS)
                printf("fipcore_write_tag_by_name - TB.PumpTemperature - %s\n", fipcore_get_error_str(ret));
            else
                printf("  OUT ID(0x8427) - TB.PumpTemperature : %f Celsius\n", TBPumpTemperature);

            TBPumpAmps = randomAmpsFloat();
            ret = fipcore_write_tag_by_name(sta1_hnd, "TB.PumpAmps", (void*) &TBPumpAmps);
            if (ret != SUCCESS)
                printf("fipcore_write_tag_by_name - TB.PumpAmps - %s\n", fipcore_get_error_str(ret));
            else
                printf("  OUT ID(0x8427) - TB.PumpAmps        : %f A\n", TBPumpAmps);

            TBPumpPressure = randomPressureUInt16();
            ret = fipcore_write_tag_by_name(sta1_hnd, "TB.PumpPressure", (void*) &TBPumpPressure);
            if (ret != SUCCESS)
                printf("fipcore_write_tag_by_name - TB.PumpPressure - %s\n", fipcore_get_error_str(ret));
            else
                printf("  OUT ID(0x8427) - TB.PumpPressure    : %d psi\n", TBPumpPressure);

            TBPumpAlarm = false;
            if (TBPumpAmps > 60) 
                TBPumpAlarm = true;
            ret = fipcore_write_tag_by_name(sta1_hnd, "TB.PumpAlarm", (void*) &TBPumpAlarm);
            if (ret != SUCCESS)
                printf("fipcore_write_tag_by_name - TB.PumpAlarm - %s\n", fipcore_get_error_str(ret));
            else
                printf("  OUT ID(0x8427) - TB.PumpAlarm       : %d\n", TBPumpAlarm);

            if (TBPumpAmps > 35) {
                TBVolume++;
                if (TBVolume > 100)
                    TBVolume = 100;
            } else {
                if (TBVolume != 0)
                    TBVolume--;
            }
            ret = fipcore_write_tag_by_name(sta1_hnd, "TB.Volume", (void*) &TBVolume);
            if (ret != SUCCESS)
                printf("fipcore_write_tag_by_name - TB.Volume - %s\n", fipcore_get_error_str(ret));
            else
                printf("  OUT ID(0x8427) - TB.Volume          : %d percent\n", TBVolume);

            // IN Tags - ID (0x8426) - Tags related to Tank A
            TAPumpTemperature = 0;
            ret = fipcore_read_tag_by_name(sta1_hnd, "TA.PumpTemperature", (void*) &TAPumpTemperature);
            if (ret != SUCCESS)
                printf("fipcore_read_tag_by_name - TA.PumpTemperature - %s\n", fipcore_get_error_str(ret));
            else
                printf("  IN  ID(0x8426) - TA.PumpTemperature : %f Celsius\n", TAPumpTemperature);

            TAPumpAmps = 0;
            ret = fipcore_read_tag_by_name(sta1_hnd, "TA.PumpAmps", (void*) &TAPumpAmps);
            if (ret != SUCCESS)
                printf("fipcore_read_tag_by_name - TA.PumpAmps - %s\n", fipcore_get_error_str(ret));
            else
                printf("  IN  ID(0x8426) - TA.PumpAmps        : %f A\n", TAPumpAmps);

            TAPumpPressure = 0;
            ret = fipcore_read_tag_by_name(sta1_hnd, "TA.PumpPressure", (void*) &TAPumpPressure);
            if (ret != SUCCESS)
                printf("fipcore_read_tag_by_name - TA.PumpPressure - %s\n", fipcore_get_error_str(ret));
            else
                printf("  IN  ID(0x8426) - TA.PumpPressure    : %d psi\n", TAPumpPressure);

            TAPumpAlarm = false;
            ret = fipcore_read_tag_by_name(sta1_hnd, "TA.PumpAlarm", (void*) &TAPumpAlarm);
            if (ret != SUCCESS)
                printf("fipcore_read_tag_by_name - TA.PumpAlarm - %s\n", fipcore_get_error_str(ret));
            else
                printf("  IN  ID(0x8426) - TA.PumpAlarm       : %d\n", TAPumpAlarm);

            TAVolume = 0;
            ret = fipcore_read_tag_by_name(sta1_hnd, "TA.Volume", (void*) &TAVolume);
            if (ret != SUCCESS)
                printf("fipcore_read_tag_by_name - TA.Volume - %s\n", fipcore_get_error_str(ret));
            else
                printf("  IN  ID(0x8426) - TA.Volume          : %d percent\n", TAVolume);

            loop_counter_1s = 200;

            printf("\n");
        }

        event_processing(sta1_hnd); 
        
        if (kbhit())
            break;
        
#if defined(_WIN32) || defined(_WIN64)
        Sleep(5);
#else
        usleep(5000);
#endif
        loop_counter_10s--;
        loop_counter_1s--;

        ret = fipcore_opcua_run_iterate(sta1_hnd, false, &opc_timeout);
        if (ret != SUCCESS)
            goto _exit;
    }

#if defined(__linux__) || defined(__APPLE__)
    changemode(0);
#endif
    
    ret = fipcore_stop_ba(sta1_hnd);

    ret = fipcore_opcua_run_shutdown(sta1_hnd);

_exit:  
    if (ret != SUCCESS)
        printf("STATION 1 - %s\n", fipcore_get_error_str(ret));

    ret = fipcore_stop_network(sta1_hnd);
    if (ret != SUCCESS)
        printf("STATION 1 - %s\n", fipcore_get_error_str(ret));
_exit1:
    ret = fipcore_device_close(sta1_hnd);
    if (ret != SUCCESS)
        printf("STATION 1 - %s\n", fipcore_get_error_str(ret));
}

void event_processing(fipcore_device_handle* hnd)
{
    uint32_t ret = SUCCESS;
    uint16_t event_code = 0;
    
    ret = fipcore_read_evt(hnd, &event_code, &event_parameter);

    while (ret == SUCCESS) { // All events are popped

        bool msgReceivedToTreat = false;

        printf("  fipcore_read_evt - ");
        switch (event_code) {
        case FIP_EVT_SEND_VAR_P:    printf("SEND_VAR_P - AccessKey: 0x%04x\n", event_parameter); break;
        case FIP_EVT_SEND_VAR_T:    printf("SEND_VAR_T - AccessKey: 0x%04x\n", event_parameter); break;
        case FIP_EVT_RECEIVE_VAR_P: printf("RECEIVE_VAR_P\n"); break;
        case FIP_EVT_RECEIVE_VAR_T: printf("RECEIVE_VAR_T\n"); break;
        case FIP_EVT_RECEIVE_MSG:   printf("RECEIVE_MSG\n"); msgReceivedToTreat = true; break;
        case FIP_EVT_SEND_MSG:      printf("SEND_MSG - Report: 0x%04x\n", event_parameter); break;
        case FIP_EVT_SEND_APU:      printf("SEND_APU\n"); break;
        case FIP_EVT_SEND_APN:      printf("SEND_APN\n"); break;
        case FIP_EVT_BA_ACTIVITY:   printf("BA_ACTIVITY\n"); break;
        case FIP_EVT_BA_STOP1:      printf("BA_STOP1\n"); break;
        case FIP_EVT_BA_STOP2:      printf("BA_STOP2\n"); break;
        case FIP_EVT_BA_STOP3:      printf("BA_STOP3\n"); break;
        case FIP_EVT_BA_IDLE:       printf("BA_IDLE\n"); break;
        default:
            if ((event_code & 0x0500) == 0x0500) {
                uint8_t suspend_number = (uint8_t) (event_code & 0x00FF);
                printf("BA_SUSPEND%d - BA Address Program: 0x%04x\n", suspend_number, event_parameter);
            } else
                printf("UNKNOWN - EventCode: 0x%04x - EventParameter: 0x%04x\n", event_code, event_parameter);          

        break;
        }

        if (msgReceivedToTreat) {
            uint32_t ret1 = SUCCESS;
            uint16_t i = 0;
            MESSAGE_FRAME_IN c_msg = {0};

            ret = fipcore_read_msg(hnd, &c_msg); 
            if (ret == SUCCESS) {

                uint8_t srcSeg  = c_msg.Src_Segment;
                uint8_t destSeg = c_msg.Dest_Segment;
                uint16_t srcId  = (((uint16_t) c_msg.Src_Add_PF) << 8)  + c_msg.Src_Add_Pf;
                uint16_t destId = (((uint16_t) c_msg.Dest_Add_PF) << 8) + c_msg.Dest_Add_Pf;

                printf("  fipcore_read_msg - From ID(0x%04x) to ID(0x%04x) - ", srcId, destId);

                printf("MSG: ");
                for (i = 0; i < c_msg.Msg_Length; i++)
                    printf("%c", c_msg.Useful_Data[i]);
                printf("\n");
            } else
                printf("ERROR - %s\n", fipcore_get_error_str(ret));
        }

        ret = fipcore_read_evt(hnd, &event_code, &event_parameter);
    }
}

void medium_processing(fipcore_device_handle* hnd)
{
    uint32_t ret = SUCCESS;
    FIP_NETWORK_STATUS  fipNetworkStatus = {0};

    // Lecture du medium
    ret = fipcore_get_medium(hnd, &fipNetworkStatus);
    if (ret != SUCCESS)
        printf("  fipcore_get_medium error: %08x", ret);
    else {
        bool ch1Active = ((fipNetworkStatus.FIELDUAL_Status & MEDIUM_STATUS_CH1_VALID) == MEDIUM_STATUS_CH1_VALID);
        bool ch2Active = ((fipNetworkStatus.FIELDUAL_Status & MEDIUM_STATUS_CH2_VALID) == MEDIUM_STATUS_CH2_VALID);
        bool ch1TXError = ((fipNetworkStatus.FIELDUAL_Status & MEDIUM_STATUS_CH1_TX_ERROR) == MEDIUM_STATUS_CH1_TX_ERROR);
        bool ch2TXError = ((fipNetworkStatus.FIELDUAL_Status & MEDIUM_STATUS_CH2_TX_ERROR) == MEDIUM_STATUS_CH2_TX_ERROR);
        bool ch1WatchdogError = ((fipNetworkStatus.FIELDUAL_Status & MEDIUM_STATUS_CH1_WATCHDOG) == MEDIUM_STATUS_CH1_WATCHDOG);
        bool ch2WatchdogError = ((fipNetworkStatus.FIELDUAL_Status & MEDIUM_STATUS_CH2_WATCHDOG) == MEDIUM_STATUS_CH2_WATCHDOG);
        char baStatusStr[255];

        switch (fipNetworkStatus.BA_Status) {
            case BA_STATUS_STOPPED:        strcpy(baStatusStr, "STOPPED"); break;
            case BA_STATUS_STARTING:       strcpy(baStatusStr, "STARTING"); break;
            case BA_STATUS_IDLE:           strcpy(baStatusStr, "IDLE"); break;
            case BA_STATUS_MACRO_END:      strcpy(baStatusStr, "MACRO END"); break;
            case BA_STATUS_SENDING_ID_DAT: strcpy(baStatusStr, "SENDING ID_DAT"); break;
            case BA_STATUS_SENDING_ID_MSG: strcpy(baStatusStr, "SENDING ID_MSG"); break;
            case BA_STATUS_TESTING:        strcpy(baStatusStr, "TESTING"); break;
            case BA_STATUS_TESTING_END:    strcpy(baStatusStr, "TESTING END"); break;
            case BA_STATUS_PENDING:        strcpy(baStatusStr, "PENDING"); break;
            case BA_STATUS_APER_WINDOW:    strcpy(baStatusStr, "APER. VAR WINDOW"); break;
            case BA_STATUS_MSG_WINDOW:     strcpy(baStatusStr, "APER. MSG WINDOW"); break;
            case BA_STATUS_WAITING_TIME:   strcpy(baStatusStr, "WAITING TIME"); break;
            case BA_STATUS_WAITING_SYNC:   strcpy(baStatusStr, "WAITING SYNC."); break;
            case BA_STATUS_WAITING_SYNC_SILENT: strcpy(baStatusStr, "WAITING SYNC. SILENT"); break;
            default: strcpy(baStatusStr, "UNKNOWN"); break;

        }

        printf("  fipcore_get_medium\n");
        printf("    No_Error_Channel1   : %d\n", fipNetworkStatus.No_Error_Channel1);
        printf("    No_Error_Channel2   : %d\n", fipNetworkStatus.No_Error_Channel2);
        printf("    Error_Channel1      : %d\n", fipNetworkStatus.Error_Channel1);
        printf("    Error_Channel2      : %d\n", fipNetworkStatus.Error_Channel2);
        printf("    Error_No_Echo       : %d\n", fipNetworkStatus.Error_No_Echo);
        printf("    Error_Testp_Channel1: %d\n", fipNetworkStatus.Error_Testp_Channel1);
        printf("    Error_Testp_Channel2: %d\n", fipNetworkStatus.Error_Testp_Channel2);
        printf("    BA_Current_Macrocyle: %04xh\n", fipNetworkStatus.BA_Current_Macrocyle);
        printf("    BA_Status           : %04xh - [%s]\n", fipNetworkStatus.BA_Status, baStatusStr);
        printf("    FIELDUAL_Status     : %04xh\n", fipNetworkStatus.FIELDUAL_Status);
        printf("          [Channel 1 - Active: %d - TX Error: %d - Watchdog Error: %d]\n", ch1Active, ch1TXError, ch1WatchdogError);
        printf("          [Channel 2 - Active: %d - TX Error: %d - Watchdog Error: %d]\n", ch2Active, ch2TXError, ch2WatchdogError);


    }
}

void print_device_info(FIP_DEVICECONF* device_conf)
{
    switch(device_conf->device_type) {
    case PCI_PCIE_TYPE:     printf("   device_type     : PCI_PCIE_TYPE \n"); break;
    case USB_TYPE:          printf("   device_type     : USB_TYPE \n"); break;
    default:                printf("   device_type     : UNKNOWN_TYPE \n"); break;
    }
    printf("   device_index    : %d \n", device_conf->device_index);
    printf("   product_number  : 0x%08x \n", device_conf->product_number);
    printf("   device_version  : %d.%d.%d \n", device_conf->device_version.major,
                                            device_conf->device_version.minor,
                                            device_conf->device_version.revision);
    switch(device_conf->fip_speed) {
    case SPEED_31K25:   printf("   fip_speed       : SPEED_31K25 \n"); break;
    case SPEED_1M:      printf("   fip_speed       : SPEED_1M \n"); break;
    case SPEED_2M5:     printf("   fip_speed       : SPEED_2M5 or SPEED_5M \n"); break;
    default:            printf("   fip_speed       : UNKNOWN_SPEED \n"); break;
    }
    switch(device_conf->fip_impedance) {
    case IMPEDANCE_150: printf("   fip_impedance   : IMPEDANCE_150 \n\n"); break;
    case IMPEDANCE_120: printf("   fip_impedance   : IMPEDANCE_120 \n\n"); break;
    default:            printf("   fip_impedance   : UNKNOWN_IMPEDANCE \n\n"); break;
    }
}

void print_conf(const FIP_USERCONF* fuc)
{
    uint16_t cp = 0;
    uint16_t i  = 0;


    printf("/*****************************************************************************/\n");
    printf("/*                              FIP_USERCONF                                 */\n");
    printf("/*****************************************************************************/\n");
    if (fuc != NULL) {
        printf("    - this_station               : \n");
        printf("        - subscriber_number      : %d \n", fuc->this_station->subscriber_number);
        printf("        - subscriber_name        : %s \n", fuc->this_station->subscriber_name);
        printf("    - speed                      : %d \n", fuc->speed);
        printf("    - t_slot                     : %d \n", fuc->t_slot);
        printf("    - silence                    : %d \n", fuc->silence);
        printf("    - turn_around                : %d \n", fuc->turn_around);
        printf("    - frame_type                 : %d \n", fuc->frame_type);
        printf("    - is_arbiter                 : %d \n", fuc->is_arbiter);
        printf("    - arbiter                    : \n");
        printf("        - ba_max_subscriber_number : %d \n", fuc->arbiter->ba_max_subscriber_number);
        printf("        - ba_priority            : %d \n", fuc->arbiter->ba_priority);
        printf("        - ba_max_priority        : %d \n", fuc->arbiter->ba_max_priority);
        printf("        - ba_optimization        : %d \n", fuc->arbiter->ba_optimization);
        printf("        - nb_ba_1_instructions   : %d \n", cp = fuc->arbiter->nb_ba_1_instructions);
        printf("        - ba_1_instructions      : \n");
        {
            INSTRUCTION* instructions = fuc->arbiter->ba_1_instructions;
            for (i = 0; i < cp; i++)
                printf("            - %s \n", (instructions + i)->op_code);
        }
        printf("        - nb_ba_2_instructions   : %d \n", cp = fuc->arbiter->nb_ba_2_instructions);
        printf("        - ba_2_instructions      : \n");
        {
            INSTRUCTION* instructions = fuc->arbiter->ba_2_instructions;
            for (i = 0; i < cp; i++)
                printf("            - %s \n", (instructions + i)->op_code);
        }   
        printf("    - nb_variables               : %d \n", cp = fuc->nb_variables);
        printf("    - variables                  : \n");
        {
            for (i = 0; i < cp; i++) {
                printf("        - number %d \n", i);
                printf("            - name               : %s \n", fuc->variables[i].name);
                printf("            - identifier         : 0x%04x \n", fuc->variables[i].identifier);
                printf("            - pdu                : 0x%04x \n", fuc->variables[i].pdu);
                printf("            - length             : %d \n", fuc->variables[i].length);
                printf("            - type               : %d \n", fuc->variables[i].type);
                printf("            - refreshment_type   : %d \n", fuc->variables[i].refreshment_type);
                printf("            - refreshment_period : 0x%04x \n", fuc->variables[i].refreshment_period);
                printf("            - promptitude_type   : %d \n", fuc->variables[i].promptitude_type);
                printf("            - promptitude_period : 0x%04x \n", fuc->variables[i].promptitude_period);
                printf("            - msg_channel        : %d \n", fuc->variables[i].msg_channel);
                printf("            - messages_service   : 0x%02x \n", fuc->variables[i].messages_service);
            }
        }
        printf("    - nb_messages                : %d \n", cp = fuc->nb_messages);
        printf("    - messages                   : \n");
        {
            for (i = 0; i < cp; i++) {
                printf("        - number %d \n", i);
                printf("            - name               : %s \n", fuc->messages[i].name);
                printf("            - type               : %d \n", fuc->messages[i].type);
                printf("            - channel            : %d \n", fuc->messages[i].channel);
                printf("            - acknowledgment     : %d \n", fuc->messages[i].acknowledgment);
                printf("            - destination_identifier: 0x%04x  \n", fuc->messages[i].destination_identifier);
                printf("            - destination_segment: 0x%02x  \n", fuc->messages[i].destination_segment);
                printf("            - source_identifier  : 0x%04x  \n", fuc->messages[i].source_identifier);
                printf("            - source_segment     : 0x%02x  \n", fuc->messages[i].source_segment);
            }
        }
        printf("     - nb_tags            : %d \n", cp = fuc->nb_tags);
        printf("     - tags               : \n");
        {
            for (i = 0; i < cp; i++) {
                printf("        - number %d \n", i);
                printf("            - tag_name      : %s \n", fuc->tags[i].tag_name);
                printf("            - tag_id        : %d \n", fuc->tags[i].tag_id);
                printf("            - type_name     : %s \n", fuc->tags[i].type_name);
                printf("            - type_id       : %d \n", fuc->tags[i].type_id);
                printf("            - access_rights : %d \n", fuc->tags[i].access_rights);
                printf("            - description   : %s \n", fuc->tags[i].description);
                printf("            - identifier    : 0x%04x \n", fuc->tags[i].identifier);
                printf("            - start_bit     : %d \n", fuc->tags[i].start_bit);
                printf("            - stop_bit      : %d \n", fuc->tags[i].stop_bit);
                printf("            - endianness    : %d \n", fuc->tags[i].endianness);
            }
        }
    } else {
        printf("    - user_conf               : \n");
        printf("        - unallocated \n");
    }
}