//  Created on : 17 Avr 2018
//  Author : M. Coroyer
//  Company : © 2014-2018 Exoligent
//  Description:
//  This example is similar to the "fipcore_rawframe" example
//  Here, rather than polling software on events from the FIP network, 
//  we use IRQs to synchronize on hardware events

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "fipcore.h"

#if defined(_WIN32) || defined(_WIN64)
#   include <windows.h>
#   include <conio.h>
#elif defined(__linux__) || defined(__APPLE__) || defined(__QNX__)
#   include <unistd.h>     // usleep()
#   include <termios.h>
#   include <sys/types.h>
#   include <sys/time.h>

void changemode(int dir)
{
    static struct termios oldt, newt;
    
    if (dir == 1) {
        tcgetattr(STDIN_FILENO, &oldt);
        newt = oldt;
        newt.c_lflag &= ~(ICANON | ECHO);
        tcsetattr(STDIN_FILENO, TCSANOW, &newt);
    } else
        tcsetattr(STDIN_FILENO, TCSANOW, &oldt);
}

int kbhit(void)
{
    struct timeval tv;
    fd_set rdfs;
    
    tv.tv_sec = 0;
    tv.tv_usec = 0;
    
    FD_ZERO(&rdfs);
    FD_SET(STDIN_FILENO, &rdfs);
    
    select(STDIN_FILENO + 1, &rdfs, NULL, NULL, &tv);
    return FD_ISSET(STDIN_FILENO, &rdfs);
}

unsigned GetTickCount()
{
    struct timeval tv;
    if(gettimeofday(&tv, NULL) != 0)
            return 0;

    return (tv.tv_sec * 1000) + (tv.tv_usec / 1000);
}
#endif

static uint32_t inc_data = 0;
static uint32_t dec_data = 0xFFFFFFFF;
uint16_t event_parameter = 0;

char* sta_xml_path[2][4] = {
    {"../xml/sta0_worldfip_31K25.xml", "../xml/sta0_worldfip_1M.xml", "../xml/sta0_worldfip_2M5.xml", "../xml/sta0_worldfip_5M.xml"},
    {"../xml/sta1_worldfip_31K25.xml", "../xml/sta1_worldfip_1M.xml", "../xml/sta1_worldfip_2M5.xml", "../xml/sta1_worldfip_5M.xml"}
};

void station0_process(uint8_t device_type, uint8_t device_index, char* xml_path);
void station1_process(uint8_t device_type, uint8_t device_index, char* xml_path);
void print_device_info(FIP_DEVICECONF* device_conf);
void print_conf(const FIP_USERCONF* fuc);
void print_stats(const FIP_STATISTICS* stats);
void print_medium(const FIP_NETWORK_STATUS* medium);
void event_processing(fipcore_device_handle* hnd, void* ctx);
void synchro_processing(fipcore_device_handle* hnd, void* ctx);

int main(int argCount, char *argValue[])
{
    uint32_t ret = SUCCESS;
    uint8_t i = 0;
    uint8_t device_type;
    uint8_t device_index;
    uint8_t station_number;
    uint8_t station_speed;
    FIP_DEVICECONF devices_list[20];
    uint8_t nb_detected_devices = 0;
    bool    is_matched = false;
    char    xml_path[255];
    char    temp[10];

    // Device Enumeration
    printf("/*****************************************************************************/\n");
    printf("/*                          DEVICES ON COMPUTER                              */\n");
    printf("/*****************************************************************************/\n");

    ret = fipcore_get_devices_number(&nb_detected_devices);
    if (ret != SUCCESS) {
        printf("fipcore_get_devices_number - %s\n", fipcore_get_error_str(ret));
        return 0;
    }

    printf("NUMBER OF DETECTED DEVICES - %d\n", nb_detected_devices);

    ret = fipcore_get_devices_list(devices_list, &nb_detected_devices);
    if (ret != SUCCESS) {
        printf("fipcore_get_devices_list - %s\n", fipcore_get_error_str(ret));
        return 0;
    }

    for (i = 0; i < nb_detected_devices; i++)
        print_device_info(&(devices_list[i]));

    if (argCount < 4) {
        printf("This application must have this form: fipcore_irq.exe 'device_type' 'device_index' 'station_number' 'station_speed' \n");
        printf("'device_type': 1: PCI_PCIE_TYPE - 2: USB_TYPE\n");
        printf("'device_index': 0 to 255\n");
        printf("'station_number': 0: Station 0 - 1: Station 1\n");
        printf("'station_speed': 0: 31.25 Kbps - 1: 1 Mbps - 2: 2.5 Mbps - 3: 5Mbps\n");
        return 0;
    }

    sscanf (argValue[1], "%s", temp);
    device_type = (uint8_t) atoi(temp);
    memset(temp, 0, sizeof(temp));

    sscanf (argValue[2], "%s", temp);
    device_index = (uint8_t) atoi(temp);
    memset(temp, 0, sizeof(temp));

    sscanf (argValue[3], "%s", temp);
    station_number = (uint8_t) atoi(temp);
    memset(temp, 0, sizeof(temp));

    sscanf (argValue[4], "%s", temp);
    station_speed = (uint8_t) atoi(temp);
    memset(temp, 0, sizeof(temp));
    
    if (!(device_type < MAX_TYPE)) {
        printf("Invalid 'device_type' parameter.\n");
        return 0;
    }
    if (station_number > 1) {
        printf("Invalid 'station_number' parameter.\n");
        return 0; 
    }
    if (station_speed > 3) {
        printf("Invalid 'station_speed' parameter.\n");
        return 0; 
    }

    for (i = 0; i < nb_detected_devices; i++) {

        if ((devices_list[i].device_type == device_type) && 
            (devices_list[i].device_index == device_index)) {

                if (devices_list[i].fip_speed != station_speed) {

                    if (!((devices_list[i].fip_speed == SPEED_2M5) && 
                        (station_speed == SPEED_5M))) {
                        // Support 2.5 Mbps device with 5 Mbps configuration

                        // else exit
                        printf("Station speed must match with device speed.\n");
                        return 0;
                    }
                }
                is_matched = true;
        }
    }

    if (is_matched) {

        if (station_number == 0)
            station0_process(device_type, device_index, sta_xml_path[station_number][station_speed]);
        else
            station1_process(device_type, device_index, sta_xml_path[station_number][station_speed]);
    } else
        printf("No device was found with device_type (%d) and device_index (%d).\n", device_type, device_index);

    return 0;
}

void station0_process(uint8_t device_type, uint8_t device_index, char* xml_path)
{
    uint32_t ret = SUCCESS;
    FIP_USERCONF      user_conf = {0};
    USERDEF_VARIABLE  var_conf = {0};
    PRODUCED_VARIABLE p_var = {0};
    MESSAGE_FRAME_OUT p_msg = {0};
    FIP_STATISTICS stats = {0};
    FIP_NETWORK_STATUS medium = {0};
    FIP_USERCALLBACKS user_callbacks = {0};
    uint8_t i, j = 0;
    uint16_t src_id;
    uint16_t dst_id;
    uint16_t msg_length;
    char msg_data[] = "Hello World\n";
    fipcore_device_handle* sta0_hnd = NULL;
    uint8_t station_number = 0;

    unsigned long tstart_10s, tstart_200ms, tcurrent;

    printf("/*****************************************************************************/\n");
    printf("/*                          STATION 0 TEST                                   */\n");
    printf("/*****************************************************************************/\n");

    sta0_hnd = fipcore_device_open(device_type, device_index, &ret);
    if (ret != SUCCESS) {
        printf("STATION 0 - %s\n", fipcore_get_error_str(ret));
        goto _exit1;
    }

    user_callbacks.fip_event = &event_processing;
    user_callbacks.fip_synchro = &synchro_processing;
    user_callbacks.ctx_ptr = (void*) &station_number;
    ret = fipcore_init_network(sta0_hnd, XML, xml_path, &user_callbacks);
    if (ret != SUCCESS)
        goto _exit;

    ret = fipcore_start_network(sta0_hnd);
    if (ret != SUCCESS)
        goto _exit;

    ret = fipcore_start_ba(sta0_hnd, 1);
    if (ret != SUCCESS)
        goto _exit;
    
    // Output variables init
    fipcore_write_var_by_id(sta0_hnd, 0x0001, &p_var);
    fipcore_write_var_by_id(sta0_hnd, 0x8426, &p_var);
    fipcore_write_var_by_id(sta0_hnd, 0x8322, &p_var);

    // Output message init
    src_id = 0x8426;
    dst_id = 0x8427;

    msg_length = MSG_HEADER_SIZE + sizeof(msg_data);
    p_msg.Msg_Mode_PF = (uint8_t) (MSG_MODE_NO_ACK >> 8);
    p_msg.Msg_Mode_Pf = (uint8_t) MSG_MODE_NO_ACK;
    p_msg.Msg_Length_PF = (uint8_t) (msg_length >> 8);
    p_msg.Msg_Length_Pf = (uint8_t) msg_length;
    //      Message header
    p_msg.Dest_Add_PF = (uint8_t) (dst_id >> 8);
    p_msg.Dest_Add_Pf = (uint8_t) dst_id;
    p_msg.Dest_Segment = 0;
    p_msg.Src_Add_PF = (uint8_t) (src_id >> 8);
    p_msg.Src_Add_Pf = (uint8_t) src_id;
    p_msg.Src_Segment = 0;
    for (i = 0; i < sizeof(msg_data); i++)
        p_msg.Useful_Data[i] = (uint8_t) msg_data[i];

#if defined(__linux__) || defined(__APPLE__)
    changemode(1);
#endif
    
    tstart_10s = GetTickCount();
    tstart_200ms = GetTickCount();

    for (;;) {
        // Get current tick
        tcurrent = GetTickCount();

        if ((tcurrent - tstart_10s) > 10000) {    // Each 10s

            tstart_10s = GetTickCount();

            // ID (0x8426) to ID (0x8427) message is sent (data: "Hello World")
            // Send on channel 0 (aperiodic)
            ret = fipcore_write_msg(sta0_hnd, 0, &p_msg); 
            if (ret != SUCCESS)
                goto _exit;

            // Aperiodic Request (ID (0x8322)) is sent
            ret = fipcore_send_aper_by_id(sta0_hnd, 0x8322, 1); // 1: Urgent Mode
            if (ret != SUCCESS)
                goto _exit;
        }

        if ((tcurrent - tstart_200ms) > 200) {    // Each 200ms

            tstart_200ms = GetTickCount();

            // Get Fip Statistics & print
            fipcore_get_stats(sta0_hnd, &stats);
            print_stats(&stats);

            // Get Medium  & print
            fipcore_get_medium(sta0_hnd, &medium);
            print_medium(&medium);
        }
        
        if (kbhit())
            break;

#if defined(_WIN32) || defined(_WIN64)
        Sleep(2);
#else
        usleep(2000);
#endif

    }

#if defined(__linux__) || defined(__APPLE__)
    changemode(0);
#endif
    
    ret = fipcore_stop_ba(sta0_hnd);

_exit:
    if (ret != SUCCESS)
        printf("STATION 0 - %s\n", fipcore_get_error_str(ret));

    ret = fipcore_stop_network(sta0_hnd);
    if (ret != SUCCESS)
        printf("STATION 0 - %s\n", fipcore_get_error_str(ret));
_exit1:
    ret = fipcore_device_close(sta0_hnd);
    if (ret != SUCCESS)
        printf("STATION 0 - %s\n", fipcore_get_error_str(ret));
}

void station1_process(uint8_t device_type, uint8_t device_index, char* xml_path)
{
    uint32_t ret = SUCCESS;
    FIP_USERCONF      user_conf = {0};
    USERDEF_VARIABLE  var_conf = {0};
    PRODUCED_VARIABLE p_var = {0};
    FIP_STATISTICS stats = {0};
    FIP_NETWORK_STATUS medium = {0};
    FIP_USERCALLBACKS user_callbacks = {0};
    uint8_t i, j = 0;
    fipcore_device_handle* sta1_hnd = NULL;
    uint8_t station_number = 1;

    unsigned long tstart_10s, tstart_200ms, tcurrent;

    printf("/*****************************************************************************/\n");
    printf("/*                          STATION 1 TEST                                   */\n");
    printf("/*****************************************************************************/\n");

    sta1_hnd = fipcore_device_open(device_type, device_index, &ret);
    if (ret != SUCCESS) {
        printf("STATION 1 - %s\n", fipcore_get_error_str(ret));
        goto _exit1;
    }

    user_callbacks.fip_event = &event_processing;
    user_callbacks.fip_synchro = &synchro_processing;
    user_callbacks.ctx_ptr = (void*) &station_number;
    ret = fipcore_init_network(sta1_hnd, XML, xml_path, &user_callbacks);
    if (ret != SUCCESS)
        goto _exit;

    ret = fipcore_start_network(sta1_hnd);
    if (ret != SUCCESS)
        goto _exit;

    ret = fipcore_start_ba(sta1_hnd, 1);
    if (ret != SUCCESS)
        goto _exit;

    // Output variables init
    fipcore_write_var_by_id(sta1_hnd, 0x8427, &p_var);

#if defined(__linux__) || defined(__APPLE__)
    changemode(1);
#endif
    
    tstart_10s = GetTickCount();
    tstart_200ms = GetTickCount();

    for (;;) {

        // Get current tick
        tcurrent = GetTickCount();

        if ((tcurrent - tstart_200ms) > 200) {    // Each 200ms

            tstart_200ms = GetTickCount();

            // Get Fip Statistics & print
            fipcore_get_stats(sta1_hnd, &stats);
            print_stats(&stats);

            // Get Medium & print
            fipcore_get_medium(sta1_hnd, &medium);
            print_medium(&medium);
 
        }
        
        if (kbhit())
            break;

#if defined(_WIN32) || defined(_WIN64)
        Sleep(2);
#else
        usleep(2000);
#endif
    }

#if defined(__linux__) || defined(__APPLE__)
    changemode(0);
#endif
    ret = fipcore_stop_ba(sta1_hnd);

_exit:  
    if (ret != SUCCESS)
        printf("STATION 1 - %s\n", fipcore_get_error_str(ret));

    ret = fipcore_stop_network(sta1_hnd);
    if (ret != SUCCESS)
        printf("STATION 1 - %s\n", fipcore_get_error_str(ret));
_exit1:
    ret = fipcore_device_close(sta1_hnd);
    if (ret != SUCCESS)
        printf("STATION 1 - %s\n", fipcore_get_error_str(ret));
}

void event_processing(fipcore_device_handle* hnd, void* ctx)
{
    uint32_t ret = SUCCESS;
    uint16_t event_code = 0;
    uint16_t id = 0;
    uint8_t direction = 0;

    ret = fipcore_read_evt(hnd, &event_code, &event_parameter);

    while (ret == SUCCESS) { // All events are popped

        bool msgReceivedToTreat = false;

        /* treat event code here */
        switch (event_code) { 
        case FIP_EVT_SEND_VAR_P:    break;
        case FIP_EVT_SEND_VAR_T:    break;
        case FIP_EVT_RECEIVE_VAR_P: break;
        case FIP_EVT_RECEIVE_VAR_T: break;
        case FIP_EVT_RECEIVE_MSG:   msgReceivedToTreat = true; break;
        case FIP_EVT_SEND_MSG:      break;
        case FIP_EVT_SEND_APU:      break;
        case FIP_EVT_SEND_APN:      break;
        case FIP_EVT_BA_ACTIVITY:   break;
        case FIP_EVT_BA_STOP1:      break;
        case FIP_EVT_BA_STOP2:      break;
        case FIP_EVT_BA_STOP3:      break;
        case FIP_EVT_BA_IDLE:       break;
        default:     
        break;
        }

        if (msgReceivedToTreat) {
            uint32_t ret1 = SUCCESS;
            uint16_t i = 0;
            MESSAGE_FRAME_IN c_msg = {0};

            ret = fipcore_read_msg(hnd, &c_msg);
            if (ret == SUCCESS) {

                uint8_t srcSeg  = c_msg.Src_Segment;
                uint8_t destSeg = c_msg.Dest_Segment;
                uint16_t srcId  = (((uint16_t) c_msg.Src_Add_PF) << 8)  + c_msg.Src_Add_Pf;
                uint16_t destId = (((uint16_t) c_msg.Dest_Add_PF) << 8) + c_msg.Dest_Add_Pf;

                // printf("MSG: ");
                // for (i = 0; i < c_msg.Msg_Length; i++)
                //     printf("%c", c_msg.Useful_Data[i]);
                // printf("\n");
            } else
                printf("ERROR - %s\n", fipcore_get_error_str(ret));
        }

        ret = fipcore_read_evt(hnd, &event_code, &event_parameter);
    }
}

void synchro_processing(fipcore_device_handle* hnd, void* ctx)
{
    uint8_t station_number = *((uint8_t*) ctx);
    PRODUCED_VARIABLE p_var = {0};
    CONSUMED_VARIABLE c_var = {0};
    bool is_prompt = false;
    bool is_refreshed = false;

    if (station_number == 0) {

        // ID (0x8426) value is incremented (4 first bytes / 12 total)
        memset(&p_var, 0, sizeof(PRODUCED_VARIABLE));
        *((uint32_t*) (p_var.Data_Var)) = inc_data;
        
        fipcore_write_var_by_id(hnd, 0x8426, &p_var);

        memset(&c_var, 0, sizeof(CONSUMED_VARIABLE));
        fipcore_read_var_by_id(hnd, 0x8427, &c_var, &is_prompt, &is_refreshed);

        // ID (0x8322) value is incremented (4 first bytes / 12 total)
        // Note: ID (0x8322) is written each 1s. However, production on the network will be effective
        // only each 10s (see fipcore_send_aper_by_id routine below).
        // This is due to the fact that ID (0x8322) is not present in the scan tables of the arbiter.
        // So it's an apriodic variable.

        memset(&p_var, 0, sizeof(PRODUCED_VARIABLE));
        *((uint32_t*) (p_var.Data_Var)) = inc_data;

        fipcore_write_var_by_id(hnd, 0x8322, &p_var);

        inc_data++;

    } else if (station_number == 1) {

        memset(&c_var, 0, sizeof(CONSUMED_VARIABLE)); 
        fipcore_read_var_by_id(hnd, 0x8426, &c_var, &is_prompt, &is_refreshed);

        // ID (0x8427) value is decremented (4 first bytes / 12 total)
        memset(&p_var, 0, sizeof(PRODUCED_VARIABLE));
        *((uint32_t*) (p_var.Data_Var)) = dec_data;

        fipcore_write_var_by_id(hnd, 0x8427, &p_var);

        memset(&c_var, 0, sizeof(CONSUMED_VARIABLE));
        fipcore_read_var_by_id(hnd, 0x8322, &c_var, &is_prompt, &is_refreshed);

        dec_data--;
    }
}

void print_device_info(FIP_DEVICECONF* device_conf)
{
    switch(device_conf->device_type) {
    case PCI_PCIE_TYPE:     printf("   device_type     : PCI_PCIE_TYPE \n"); break;
    case USB_TYPE:          printf("   device_type     : USB_TYPE \n"); break;
    default:                printf("   device_type     : UNKNOWN_TYPE \n"); break;
    }
    printf("   device_index    : %d \n", device_conf->device_index);
    printf("   product_number  : 0x%08x \n", device_conf->product_number);
    printf("   device_version  : %d.%d.%d \n", device_conf->device_version.major,
                                            device_conf->device_version.minor,
                                            device_conf->device_version.revision);
    switch(device_conf->fip_speed) {
    case SPEED_31K25:   printf("   fip_speed       : SPEED_31K25 \n"); break;
    case SPEED_1M:      printf("   fip_speed       : SPEED_1M \n"); break;
    case SPEED_2M5:     printf("   fip_speed       : SPEED_2M5 or SPEED_5M \n"); break;
    default:            printf("   fip_speed       : UNKNOWN_SPEED \n"); break;
    }
    switch(device_conf->fip_impedance) {
    case IMPEDANCE_150: printf("   fip_impedance   : IMPEDANCE_150 \n\n"); break;
    case IMPEDANCE_120: printf("   fip_impedance   : IMPEDANCE_120 \n\n"); break;
    default:            printf("   fip_impedance   : UNKNOWN_IMPEDANCE \n\n"); break;
    }
}


void print_conf(const FIP_USERCONF* fuc)
{
    uint16_t cp = 0;
    uint16_t i  = 0;


    printf("/*****************************************************************************/\n");
    printf("/*                              FIP_USERCONF                                 */\n");
    printf("/*****************************************************************************/\n");
    if (fuc != NULL) {
        printf("    - this_station               : \n");
        printf("        - subscriber_number      : %d \n", fuc->this_station->subscriber_number);
        printf("        - subscriber_name        : %s \n", fuc->this_station->subscriber_name);
        printf("    - speed                      : %d \n", fuc->speed);
        printf("    - t_slot                     : %d \n", fuc->t_slot);
        printf("    - silence                    : %d \n", fuc->silence);
        printf("    - turn_around                : %d \n", fuc->turn_around);
        printf("    - frame_type                 : %d \n", fuc->frame_type);
        printf("    - is_arbiter                 : %d \n", fuc->is_arbiter);
        printf("    - arbiter                    : \n");
        printf("        - ba_max_subscriber_number : %d \n", fuc->arbiter->ba_max_subscriber_number);
        printf("        - ba_priority            : %d \n", fuc->arbiter->ba_priority);
        printf("        - ba_max_priority        : %d \n", fuc->arbiter->ba_max_priority);
        printf("        - ba_optimization        : %d \n", fuc->arbiter->ba_optimization);
        printf("        - nb_ba_1_instructions   : %d \n", cp = fuc->arbiter->nb_ba_1_instructions);
        printf("        - ba_1_instructions      : \n");
        {
            INSTRUCTION* instructions = fuc->arbiter->ba_1_instructions;
            for (i = 0; i < cp; i++)
                printf("            - %s \n", (instructions + i)->op_code);
        }
        printf("        - nb_ba_2_instructions   : %d \n", cp = fuc->arbiter->nb_ba_2_instructions);
        printf("        - ba_2_instructions      : \n");
        {
            INSTRUCTION* instructions = fuc->arbiter->ba_2_instructions;
            for (i = 0; i < cp; i++)
                printf("            - %s \n", (instructions + i)->op_code);
        }   
        printf("    - nb_variables               : %d \n", cp = fuc->nb_variables);
        printf("    - variables                  : \n");
        {
            for (i = 0; i < cp; i++) {
                printf("        - number %d \n", i);
                printf("            - name               : %s \n", fuc->variables[i].name);
                printf("            - identifier         : 0x%04x \n", fuc->variables[i].identifier);
                printf("            - pdu                : 0x%04x \n", fuc->variables[i].pdu);
                printf("            - length             : %d \n", fuc->variables[i].length);
                printf("            - type               : %d \n", fuc->variables[i].type);
                printf("            - refreshment_type   : %d \n", fuc->variables[i].refreshment_type);
                printf("            - refreshment_period : 0x%04x \n", fuc->variables[i].refreshment_period);
                printf("            - promptitude_type   : %d \n", fuc->variables[i].promptitude_type);
                printf("            - promptitude_period : 0x%04x \n", fuc->variables[i].promptitude_period);
                printf("            - msg_channel        : %d \n", fuc->variables[i].msg_channel);
                printf("            - messages_service   : 0x%02x \n", fuc->variables[i].messages_service);
            }
        }
        printf("    - nb_messages                : %d \n", cp = fuc->nb_messages);
        printf("    - messages                   : \n");
        {
            for (i = 0; i < cp; i++) {
                printf("        - number %d \n", i);
                printf("            - name               : %s \n", fuc->messages[i].name);
                printf("            - type               : %d \n", fuc->messages[i].type);
                printf("            - channel            : %d \n", fuc->messages[i].channel);
                printf("            - acknowledgment     : %d \n", fuc->messages[i].acknowledgment);
                printf("            - destination_identifier: 0x%04x  \n", fuc->messages[i].destination_identifier);
                printf("            - destination_segment: 0x%02x  \n", fuc->messages[i].destination_segment);
                printf("            - source_identifier  : 0x%04x  \n", fuc->messages[i].source_identifier);
                printf("            - source_segment     : 0x%02x  \n", fuc->messages[i].source_segment);
            }
        }
        printf("     - nb_tags            : %d \n", cp = fuc->nb_tags);
        printf("     - tags               : \n");
        {
            for (i = 0; i < cp; i++) {
                printf("        - number %d \n", i);
                printf("            - tag_name      : %s \n", fuc->tags[i].tag_name);
                printf("            - tag_id        : %d \n", fuc->tags[i].tag_id);
                printf("            - type_name     : %s \n", fuc->tags[i].type_name);
                printf("            - type_id       : %d \n", fuc->tags[i].type_id);
                printf("            - access_rights : %d \n", fuc->tags[i].access_rights);
                printf("            - description   : %s \n", fuc->tags[i].description);
                printf("            - identifier    : 0x%04x \n", fuc->tags[i].identifier);
                printf("            - start_bit     : %d \n", fuc->tags[i].start_bit);
                printf("            - stop_bit      : %d \n", fuc->tags[i].stop_bit);
                printf("            - endianness    : %d \n", fuc->tags[i].endianness);
            }
        }
    } else {
        printf("    - user_conf               : \n");
        printf("        - unallocated \n");
    }
}

void print_stats(const FIP_STATISTICS* stats)
{
    float cycle_time_us = 0;

#if defined(_WIN32) || defined(_WIN64)
    system("cls");
#else
    system("clear");
#endif

    cycle_time_us = stats->Cycle_Time / 1000000.;

    printf("/*****************************************************************************/\n");
    printf("/*                            FIP_STATISTICS                                 */\n");
    printf("/*****************************************************************************/\n");
    printf("Note 1: IRQ Non Handled:\n");
    printf("        Don't worry too much about this indicator. It indicates the number of IRQs\n");
    printf("        produced on the same IRQ line than the FIP device but not produced by it.\n\n");
    printf("Note 2: Macrocycles Counter:\n");
    printf("        The counter is clocked on the synchronization variable 0x9003.\n");
    printf("        Therefore this variable must be present in the bus arbiter's macrocycle.\n\n");
    printf("Note 3: Variables, Messages, Aperiodics List, Bus Arbiter event counters:\n");
    printf("        These counters are automatically incremented when the user calls\n");
    printf("        'fipcore_read_evt' function.\n\n");
    printf("    - IRQ Handled              : %u\n", stats->Irq_Handled_Count);
    printf("    - IRQ Non Handled          : %u\n", stats->Irq_Non_Handled_Count);
    printf("    - Macrocycles Count        : %lu\n", stats->Cycle_Count);
    printf("    - Macrocycle Time          : %.1f ms\n", cycle_time_us);
    printf("    - Variables with 'Permanent' Event\n");
    printf("        - Sent                 : %lu\n", stats->Var_P_Sent_Count);
    printf("        - Received             : %lu\n", stats->Var_P_Received_Count);
    printf("    - Variables with 'Temporary' Event\n");
    printf("        - Sent                 : %lu\n", stats->Var_T_Sent_Count);
    printf("        - Received             : %lu\n", stats->Var_T_Received_Count);
    printf("    - Messages Event\n");
    printf("        - Sent                 : %lu\n", stats->Msg_Sent_Count);
    printf("        - Received             : %lu\n", stats->Msg_Received_Count);
    printf("    - Aperiodics List Event\n");
    printf("        - Urgent Sent          : %lu\n", stats->APU_List_Sent_Count);
    printf("        - Normal Sent          : %lu\n", stats->APN_List_Sent_Count);
    printf("    - Bus Arbiter Event\n");
    printf("        - Active               : %lu\n", stats->BA_Activity_Count);
    printf("        - Idle                 : %lu\n", stats->BA_Idle_Count);
    printf("        - Suspended            : %lu\n", stats->BA_Suspend_Count);
    printf("        - Stopped On Timeout   : %lu\n", stats->BA_Stop_On_Timeout_Count);
    printf("        - Stopped On Anomalies : %lu\n", stats->BA_Stop_On_Anomalies_Count);
    printf("        - Stopped By User      : %lu\n", stats->BA_Stop_By_User_Count);
    printf("\n");

}

void print_medium(const FIP_NETWORK_STATUS* medium)
{
    bool ch1Active = ((medium->FIELDUAL_Status & MEDIUM_STATUS_CH1_VALID) == MEDIUM_STATUS_CH1_VALID);
    bool ch2Active = ((medium->FIELDUAL_Status & MEDIUM_STATUS_CH2_VALID) == MEDIUM_STATUS_CH2_VALID);
    bool ch1TXError = ((medium->FIELDUAL_Status & MEDIUM_STATUS_CH1_TX_ERROR) == MEDIUM_STATUS_CH1_TX_ERROR);
    bool ch2TXError = ((medium->FIELDUAL_Status & MEDIUM_STATUS_CH2_TX_ERROR) == MEDIUM_STATUS_CH2_TX_ERROR);
    bool ch1WatchdogError = ((medium->FIELDUAL_Status & MEDIUM_STATUS_CH1_WATCHDOG) == MEDIUM_STATUS_CH1_WATCHDOG);
    bool ch2WatchdogError = ((medium->FIELDUAL_Status & MEDIUM_STATUS_CH2_WATCHDOG) == MEDIUM_STATUS_CH2_WATCHDOG);
    char baStatusStr[255];

    printf("/*****************************************************************************/\n");
    printf("/*                            MEDIUM                                         */\n");
    printf("/*****************************************************************************/\n");

    switch (medium->BA_Status) {
        case BA_STATUS_STOPPED:        strcpy(baStatusStr, "STOPPED"); break;
        case BA_STATUS_STARTING:       strcpy(baStatusStr, "STARTING"); break;
        case BA_STATUS_IDLE:           strcpy(baStatusStr, "IDLE"); break;
        case BA_STATUS_MACRO_END:      strcpy(baStatusStr, "MACRO END"); break;
        case BA_STATUS_SENDING_ID_DAT: strcpy(baStatusStr, "SENDING ID_DAT"); break;
        case BA_STATUS_SENDING_ID_MSG: strcpy(baStatusStr, "SENDING ID_MSG"); break;
        case BA_STATUS_TESTING:        strcpy(baStatusStr, "TESTING"); break;
        case BA_STATUS_TESTING_END:    strcpy(baStatusStr, "TESTING END"); break;
        case BA_STATUS_PENDING:        strcpy(baStatusStr, "PENDING"); break;
        case BA_STATUS_APER_WINDOW:    strcpy(baStatusStr, "APER. VAR WINDOW"); break;
        case BA_STATUS_MSG_WINDOW:     strcpy(baStatusStr, "APER. MSG WINDOW"); break;
        case BA_STATUS_WAITING_TIME:   strcpy(baStatusStr, "WAITING TIME"); break;
        case BA_STATUS_WAITING_SYNC:   strcpy(baStatusStr, "WAITING SYNC."); break;
        case BA_STATUS_WAITING_SYNC_SILENT: strcpy(baStatusStr, "WAITING SYNC. SILENT"); break;
        default: strcpy(baStatusStr, "UNKNOWN"); break;

    }

    printf("    - No_Error_Channel1   : %d\n", medium->No_Error_Channel1);
    printf("    - No_Error_Channel2   : %d\n", medium->No_Error_Channel2);
    printf("    - Error_Channel1      : %d\n", medium->Error_Channel1);
    printf("    - Error_Channel2      : %d\n", medium->Error_Channel2);
    printf("    - Error_No_Echo       : %d\n", medium->Error_No_Echo);
    printf("    - Error_Testp_Channel1: %d\n", medium->Error_Testp_Channel1);
    printf("    - Error_Testp_Channel2: %d\n", medium->Error_Testp_Channel2);
    printf("    - BA_Current_Macrocyle: %04xh\n", medium->BA_Current_Macrocyle);
    printf("    - BA_Status           : %04xh - [%s]\n", medium->BA_Status, baStatusStr);
    printf("    - FIELDUAL_Status     : %04xh\n", medium->FIELDUAL_Status);
    printf("          [Channel 1 - Active: %d - TX Error: %d - Watchdog Error: %d]\n", ch1Active, ch1TXError, ch1WatchdogError);
    printf("          [Channel 2 - Active: %d - TX Error: %d - Watchdog Error: %d]\n", ch2Active, ch2TXError, ch2WatchdogError);
    printf("\n");
}