-- Copyright (C) 2014-2019 EXOLIGENT (www.exoligent.com)
-- Author: Maxime Coroyer <maxime.coroyer@exoligent.com>
-- Info: Lua dissector for FiWiP - FIP Watcher over IP Protocol
-- Version: 1.0.0 (compatible FWP v1.0)
--
-- Plugin Installation:
--      1- Set the UDP_PORT parameter below to match with the 
--         settings of your FiWiP device, and save it
--      2- Open the Wireshark HMI
--      3- Go to Help -> About Wireshark
--      4- Click on the "Folder" tab
--      5- Copy this plugin into the folder corresponding to 
--         "Personal Lua Plugin"
--      6- Restart the Wireshark HMI, the plugin is then active
-- Example: GNU/Linux
--  cp fwp.lua ~/.local/lib/wireshark/plugins/
-- ******************************************************************
local UDP_PORT = 57117
-- ******************************************************************

-- declare the protocol
fwp_proto = Proto("fwp", "FIP Watcher over IP Protocol [FWP]")

-- plugin infos
local fwp_info = {
    version = "1.0.0",
    description = "FIP Watcher over IP Protocol [FWP]",
    author = "Maxime Coroyer <maxime.coroyer@exoligent.com>"
}
set_plugin_info(fwp_info)

-- declare the value strings
local vs_evt_type = {[0xc0]="FIP Frame Received", [0xc1]="System LifeSign"}
local vs_act_status = {[0]="OFF", [1]="ON"}
local vs_rx_ch_status = {[0]="Channel 1", [1]="Channel 2"}
local vs_hdl_counter_status = {[0]="OK", [1]="NOK: Overflow"}
local vs_symbol_validity = {[0]="Valid", [1]="Invalid"}
local vs_symbol_presence = {[0]="YES", [1]="NO" }
local vs_fip_net_type = {[0]="Unknown", [1]="FIP", [2]="WorldFIP" }
local vs_fip_net_bitrate = {[0]="Unknown", [1]="31.25kbps", [2]="1Mbps", 
        [3]="2.5Mbps", [4]="5Mbps", [5]="25Mbps"}
local vs_fip_frm_ctrl = {
    [0x03]="ID_DAT",
    [0x05]="ID_MSG",
    [0x29]="ID_RQ1",
    [0x09]="ID_RQ2",
    [0x02]="RP_DAT",
    [0x06]="RP_DAT_MSG",
    [0x2a]="RP_DAT_RQ1",
    [0x0a]="RP_DAT_RQ2",
    [0x2e]="RP_DAT_RQ1_MSG",
    [0x0e]="RP_DAT_RQ2_MSG",
    [0x28]="RP_RQ1",
    [0x08]="RP_RQ2",
    [0x04]="RP_MSG_NOACK",
    [0x14]="RP_MSG_ACKe",
    [0x94]="RP_MSG_ACKo",
    [0x30]="RP_ACKpe",
    [0xb0]="RP_ACKpo",
    [0x10]="RP_ACKme",
    [0x90]="RP_ACKmo",
    [0x40]="RP_FIN "
}
local vs_fip_frm_ctrl_dir = {[0]="Response Frame", [1]="Request Frame"}
local vs_is_active = {[0]="NO", [1]="YES" }
local vs_parity = {[0]="Odd", [1]="Even"}
local vs_fip_frm_pdu_type = {[0x40] = "Compact Value", [0x50] = "Presence",
        [0x52] = "Identification"}

-- declare the fields
--  protocol field: general
local f_prot_id = ProtoField.uint16("fwp.prot_id", "Identifier", base.HEX)
local f_prot_ver = ProtoField.uint16("fwp.prot_ver", "Version", base.HEX)
local f_prot_cnt = ProtoField.uint64("fwp.pkt_cnt", "Sequence Number", base.DEC)
local f_prot_nb_evt = ProtoField.uint32("fwp.nb_evt", "Number of Events", base.DEC)
--  event field: general
local f_evt_type = ProtoField.uint8("fwp.evt_type", "Type", base.HEX, vs_evt_type)
local f_evt_len = ProtoField.uint8("fwp.evt_len", "Length", base.DEC)
--  event field: system monitoring (lifesign)
local f_sys_cnt = ProtoField.uint64("fwp.sys.lifesign_cnt", "Lifesign Number", base.DEC)
local f_sys_ep = ProtoField.uint64("fwp.sys.lifesign_epoch", "Capture Epoch Time", 
                base.DEC)
local f_sys_sn = ProtoField.uint32("fwp.sys.hw_sn", "Product Number", base.HEX)
local f_cpu_load = ProtoField.float("fwp.sys.cpu_load", "Load", base.DEC)
local f_cpu_temp = ProtoField.float("fwp.sys.cpu_temp", "Temp", base.DEC)
local f_mmc_total = ProtoField.uint32("fwp.sys.mmc_total_kb", "Total", base.DEC)
local f_mmc_used = ProtoField.uint32("fwp.sys.mmc_used_kb", "Used ", base.DEC)
local f_mmc_avail = ProtoField.uint32("fwp.sys.mmc_avail_kb", "Avail", base.DEC)
local f_mmc_used_per = ProtoField.float("fwp.sys.mmc_used_per", "Used ", base.DEC)
local f_ram_total = ProtoField.uint32("fwp.sys.ram_total_kb", "Total", base.DEC)
local f_ram_used = ProtoField.uint32("fwp.sys.ram_used_kb", "Used ", base.DEC)
local f_ram_avail = ProtoField.uint32("fwp.sys.ram_avail_kb", "Avail", base.DEC)
local f_ram_used_per = ProtoField.float("fwp.sys.ram_used_per", "Used ", base.DEC)
local f_buf_total = ProtoField.uint32("fwp.sys.buf_total_kb", "Total", base.DEC)
local f_buf_used = ProtoField.uint32("fwp.sys.buf_used_kb", "Used ", base.DEC)
local f_buf_avail = ProtoField.uint32("fwp.sys.buf_avail_kb", "Avail", base.DEC)
local f_buf_used_per = ProtoField.float("fwp.sys.buf_used_per", "Used ", base.DEC)
local f_udp_is_on = ProtoField.uint8("fwp.udp.is_on", "Status", base.DEC, vs_act_status)
local f_udp_total_on = ProtoField.uint64("fwp.udp.total_on_us", "Total ON", base.DEC)
local f_udp_total_off = ProtoField.uint64("fwp.udp.total_off_us", "Total OFF", base.DEC)
local f_udp_last_start_ep = ProtoField.uint64("fwp.udp.last_start_epoch", 
                            "Last Start Epoch Time", base.DEC)
local f_udp_last_stop_ep = ProtoField.uint64("fwp.udp.last_stop_epoch", 
                            "Last Stop Epoch Time", base.DEC)
local f_udp_last_on = ProtoField.uint64("fwp.udp.last_on_us", "Last ON", base.DEC)
local f_udp_last_off = ProtoField.uint64("fwp.udp.last_off_us", "Last OFF", base.DEC)
local f_udp_loss_cnt = ProtoField.uint16("fwp.udp.loss_cnt", "Loss Count", base.DEC)
local f_udp_rx_f_count = ProtoField.uint64("fwp.udp.rx_f_count", "Frame Total", base.DEC)
local f_udp_rx_f_rate = ProtoField.uint32("fwp.udp.rx_f_rate", "Frame Rate", base.DEC)
local f_udp_rx_b_rate = ProtoField.uint64("fwp.udp.rx_b_rate", "Byte Rate", base.DEC)
local f_udp_tx_f_count = ProtoField.uint64("fwp.udp.tx_f_count", "Frame Total", base.DEC)
local f_udp_tx_f_rate = ProtoField.uint32("fwp.udp.tx_f_rate", "Frame Rate", base.DEC)
local f_udp_tx_b_rate = ProtoField.uint64("fwp.udp.tx_b_rate", "Byte Rate", base.DEC)
local f_fip_is_on = ProtoField.uint8("fwp.fip.is_on", "Status", base.DEC, vs_act_status)
local f_fip_total_on = ProtoField.uint64("fwp.fip.total_on_us", "Total Activity",
                        base.DEC)
local f_fip_total_off = ProtoField.uint64("fwp.fip.total_off_us", "Total Inactivity", 
                        base.DEC)
local f_fip_last_start_ep = ProtoField.uint64("fwp.fip.last_start_epoch", 
                            "Last Start Epoch Time", base.DEC)
local f_fip_last_stop_ep = ProtoField.uint64("fwp.fip.last_stop_epoch", 
                            "Last Stop Epoch Time", base.DEC)
local f_fip_last_on = ProtoField.uint64("fwp.fip.last_on_us", "Last Activity", base.DEC)
local f_fip_last_off = ProtoField.uint64("fwp.fip.last_off_us", "Last Inactivity", 
                        base.DEC)
local f_fip_loss_cnt = ProtoField.uint16("fwp.fip.loss_cnt", "Loss Count", base.DEC)
local f_fip_rx_f_count = ProtoField.uint64("fwp.fip.rx_f_count", "Frame Total", base.DEC)
local f_fip_rx_f_rate = ProtoField.uint32("fwp.fip.rx_f_rate", "Frame Rate", base.DEC)
local f_fip_rx_b_rate = ProtoField.uint64("fwp.fip.rx_b_rate", "Byte Rate", base.DEC)
local f_fip_net_type = ProtoField.uint8("fwp.fip.net_type", "Network Type", base.DEC, 
                        vs_fip_net_type)
local f_fip_net_bitrate = ProtoField.uint8("fwp.fip.net_bitrate", "Network Bitrate", 
                            base.DEC, vs_fip_net_bitrate)
local f_fip_err_phy_pre = ProtoField.uint64("fwp.fip.err_phy_pre", "PRE", base.DEC)
local f_fip_err_phy_fsd = ProtoField.uint64("fwp.fip.err_phy_fsd", "FSD", base.DEC)
local f_fip_err_phy_fed = ProtoField.uint64("fwp.fip.err_phy_fed", "FED", base.DEC)
local f_fip_err_dl_ctl = ProtoField.uint64("fwp.fip.err_dl_ctl", "CTL", base.DEC)
local f_fip_err_dl_fcs = ProtoField.uint64("fwp.fip.err_dl_fcs", "FCS", base.DEC)
local f_fip_err_app_pdu_type = ProtoField.uint64("fwp.fip.err_app_pdu_type", "PDU_TYPE",
                            base.DEC)
local f_fip_err_app_pdu_len = ProtoField.uint64("fwp.fip.err_app_pdu_len", "PDU_LEN", 
                                base.DEC)
local f_fip_err_hdl_glc = ProtoField.uint64("fwp.fip.err_hdl_glc", "GLC", base.DEC)
local f_fip_err_hdl_ifc = ProtoField.uint64("fwp.fip.err_hdl_ifc", "IFC", base.DEC)
local f_fip_err_hdl_fc = ProtoField.uint64("fwp.fip.err_hdl_fc", "FC ", base.DEC)
--  event fields: fip frame received
local f_fip_frm_status = ProtoField.uint16("fwp.fip.frm_status", "Frame Status", 
        base.HEX)
local f_fip_frm_status_net_bitrate = ProtoField.uint16("fwp.fip.frm_status.bitrate", 
                                "Frame Bitrate", nil, vs_fip_net_bitrate, 0xE000)
local f_fip_frm_status_net_type = ProtoField.uint16("fwp.fip.frm_status.type", 
                                "Frame Type", nil, vs_fip_net_type, 0x1800)
local f_fip_frm_status_ch_rx = ProtoField.uint16("fwp.fip.frm_status.ch_rx", 
                            "Rx Channel", nil, vs_rx_ch_status, 0x0400)
local f_fip_frm_status_err_glc = ProtoField.uint16("fwp.fip.frm_status.err_hdl_glc", 
                            "HDL Glitch/Noise Counter Error", nil, 
                            vs_hdl_counter_status, 0x0040)
local f_fip_frm_status_err_ifc = ProtoField.uint16("fwp.fip.frm_status.err_hdl_ifc", 
                            "HDL Inter-Frame Time Counter Error", nil, 
                            vs_hdl_counter_status, 0x0020)
local f_fip_frm_status_err_fc = ProtoField.uint16("fwp.fip.frm_status.err_hdl_fc", 
                            "HDL Frame Time Counter Error", nil, 
                            vs_hdl_counter_status, 0x0010)
local f_fip_frm_status_err_fed = ProtoField.uint16("fwp.fip.frm_status.err_phy_fed", 
                            "FED: Frame End Delimiter Error", nil, 
                            vs_symbol_validity, 0x0008)
local f_fip_frm_status_err_fsd = ProtoField.uint16("fwp.fip.frm_status.err_phy_fsd", 
                            "FSD: Frame Start Delimiter Error", nil, 
                            vs_symbol_validity, 0x0004)
local f_fip_frm_status_err_no_fsd = ProtoField.uint16("fwp.fip.frm_status.err_phy_no_fsd", 
                            "FSD Presence", nil, vs_symbol_presence, 0x0002)
local f_fip_frm_status_err_pre = ProtoField.uint16("fwp.fip.frm_status.err_phy_pre", 
                            "PRE: Preamble Error", nil, vs_symbol_validity, 0x0001)

local f_fip_frm_cnt = ProtoField.uint64("fwp.fip.frm_cnt", "Frame Number", base.DEC)
local f_fip_frm_ep = ProtoField.uint64("fwp.fip.frm_epoch", "Capture Epoch Time", 
                        base.DEC)
local f_fip_frm_dur = ProtoField.uint32("fwp.fip.frm_dur", "Frame Time (unit:1/10us)", 
                        base.DEC)
local f_fip_inter_frm_dur = ProtoField.uint32("fwp.fip.inter_frm_dur", 
                            "Previous Inter-Frame Time (unit:1/10us)", base.DEC)
local f_fip_frm_glitch_cnt = ProtoField.uint16("fwp.fip.frm_glitch_cnt", 
                            "Glitches Number", base.DEC)
local f_fip_frm_jitter = ProtoField.uint8("fwp.fip.frm_jitter", 
                        "Percent Jitter (unit:0.5%)", base.DEC)
local f_fip_frm = ProtoField.bytes("fwp.fip.frm", "FIP Frame", base.SPACE)
local f_fip_frm_ctrl = ProtoField.uint8("fwp.fip.frm_ctrl", "Link Control", base.HEX, 
                        vs_fip_frm_ctrl)
local f_fip_frm_ctrl_dir = ProtoField.uint8("fwp.fip.frm_ctrl.dir", "Direction", nil, 
                        vs_fip_frm_ctrl_dir, 0x01)
local f_fip_frm_ctrl_var = ProtoField.uint8("fwp.fip.frm_ctrl.var", "Variable Transfer", 
                        nil, vs_is_active, 0x02)
local f_fip_frm_ctrl_msg = ProtoField.uint8("fwp.fip.frm_ctrl.msg", "Message Transfer", 
                        nil, vs_is_active, 0x04)
local f_fip_frm_ctrl_var_list = ProtoField.uint8("fwp.fip.frm_ctrl.var_list", 
                        "Aperiodic List Transfer", nil, vs_is_active, 0x08)
local f_fip_frm_ctrl_ack_frm = ProtoField.uint8("fwp.fip.frm_ctrl.ack", 
                            "Message Ack Frame", nil, vs_is_active, 0x10)
local f_fip_frm_ctrl_prio_or_ack = ProtoField.uint8("fwp.fip.frm_ctrl.priority", 
                        "Aperiodic List Priority (0:normal,1:urgent) \
            Message Ack State (0:negative,1:positive)", base.DEC, nil, 0x20)
local f_fip_frm_ctrl_end = ProtoField.uint8("fwp.fip.frm_ctrl.end", 
                        "End Transaction Frame", nil, vs_is_active, 0x40)
local f_fip_frm_ctrl_parity = ProtoField.uint8("fwp.fip.frm_ctrl.parity", 
                        "Message Parity", nil, vs_parity, 0x80)
local f_fip_frm_id = ProtoField.uint16("fwp.fip.frm_id", "Identifier", base.HEX)
local f_fip_frm_pdu = ProtoField.uint16("fwp.fip.frm_pdu", "Protocol Data Units", 
                        base.HEX)
local f_fip_frm_pdu_type = ProtoField.uint8("fwp.fip.frm_pdu_type", "Type", base.HEX, 
                            vs_fip_frm_pdu_type)
local f_fip_frm_pdu_len = ProtoField.uint8("fwp.fip.frm_pdu_len", "Length", base.DEC)
local f_fip_frm_user_data = ProtoField.bytes("fwp.fip.frm_user_data", "User Data", 
                            base.SPACE)
local f_fip_frm_list_ids = ProtoField.bytes("fwp.fip.frm_list_ids", "List of Identifiers", 
                            base.SPACE)
local f_fip_frm_dst_add = ProtoField.uint32("fwp.fip.frm_dst_add", "Destination", 
                            base.HEX)
local f_fip_frm_src_add = ProtoField.uint32("fwp.fip.frm_src_add", "Source", 
                            base.HEX)
local f_fip_frm_fcs = ProtoField.uint16("fwp.fip.frm_fcs", "Frame Check Sequence", 
                        base.HEX)

fwp_proto.fields = { f_prot_id, f_prot_ver, f_prot_cnt, f_prot_nb_evt, 
                    f_evt_type, f_evt_len,
                    -- event: system monitoring
                    f_sys_cnt, f_sys_ep, f_sys_sn, f_cpu_load, f_cpu_temp,
                    f_mmc_total, f_mmc_used, f_mmc_avail, f_mmc_used_per,
                    f_ram_total, f_ram_used, f_ram_avail, f_ram_used_per,
                    f_buf_total, f_buf_used, f_buf_avail, f_buf_used_per,
                    f_udp_is_on, f_udp_total_on, f_udp_total_off,
                    f_udp_last_start_ep, f_udp_last_stop_ep, f_udp_last_on,
                    f_udp_last_off, f_udp_loss_cnt, f_udp_rx_f_count, 
                    f_udp_rx_f_rate, f_udp_rx_b_rate, f_udp_tx_f_count, f_udp_tx_f_rate,
                    f_udp_tx_b_rate,
                    f_fip_is_on, f_fip_total_on, f_fip_total_off, 
                    f_fip_last_start_ep, f_fip_last_stop_ep, f_fip_last_on,
                    f_fip_last_off, f_fip_loss_cnt, f_fip_rx_f_count, 
                    f_fip_rx_f_rate, f_fip_rx_b_rate, f_fip_net_type, f_fip_net_bitrate,
                    f_fip_err_phy_pre, f_fip_err_phy_fsd, f_fip_err_phy_fed, 
                    f_fip_err_dl_ctl, f_fip_err_dl_fcs, f_fip_err_app_pdu_type, 
                    f_fip_err_app_pdu_len, f_fip_err_hdl_glc, f_fip_err_hdl_ifc, 
                    f_fip_err_hdl_fc,
                    -- event: fip frame received
                    f_fip_frm_status, f_fip_frm_status_net_bitrate, 
                    f_fip_frm_status_net_type, f_fip_frm_status_ch_rx, 
                    f_fip_frm_status_err_glc, f_fip_frm_status_err_ifc, 
                    f_fip_frm_status_err_fc, f_fip_frm_status_err_fed, 
                    f_fip_frm_status_err_fsd, f_fip_frm_status_err_no_fsd, 
                    f_fip_frm_status_err_pre, f_fip_frm_cnt, f_fip_frm_ep, f_fip_frm_dur,
                    f_fip_inter_frm_dur, f_fip_frm_glitch_cnt, f_fip_frm_jitter, 
                    f_fip_frm, f_fip_frm_ctrl, 
                    f_fip_frm_ctrl_dir, f_fip_frm_ctrl_var, f_fip_frm_ctrl_msg, 
                    f_fip_frm_ctrl_var_list, f_fip_frm_ctrl_ack_frm, 
                    f_fip_frm_ctrl_prio_or_ack, f_fip_frm_ctrl_end, 
                    f_fip_frm_ctrl_parity, f_fip_frm_id, f_fip_frm_pdu, 
                    f_fip_frm_pdu_type, f_fip_frm_pdu_len, f_fip_frm_user_data, 
                    f_fip_frm_list_ids, f_fip_frm_dst_add, f_fip_frm_src_add, 
                    f_fip_frm_fcs }

-- create the dissection function
function fwp_proto.dissector(buffer, pinfo, tree)

    -- set the protocol column
    pinfo.cols['protocol'] = "FWP"

    -- create the FWP protocol tree item
    local t_fwp = tree:add(fwp_proto, buffer())
    local offset = 0

    -- fwp header
    local b_prot_hdr = buffer(offset, 16)
    local b_prot_id = buffer(offset, 2)
    local b_prot_ver = buffer(offset + 2, 2)
    local b_prot_cnt = buffer(offset + 4, 8)
    local b_prot_nb_evt = buffer(offset + 12, 4)

    local nb_evt = b_prot_nb_evt:uint()

    local t_prot_hdr = t_fwp:add(b_prot_hdr, "FWP Header:"):append_text(
                        " Sequence Num: " .. b_prot_cnt:uint64() ..
                        " , Number of events: " .. b_prot_nb_evt:uint64())
    t_prot_hdr:add(f_prot_id, b_prot_id)
    t_prot_hdr:add(f_prot_ver, b_prot_ver)
    t_prot_hdr:add(f_prot_cnt, b_prot_cnt)
    t_prot_hdr:add(f_prot_nb_evt, b_prot_nb_evt)

    offset = offset + 16

    local evt_num = 0

    local function evt_pass(buffer)
        local b_evt_hdr = buffer(offset, 3)
        local b_evt_type = buffer(offset, 1)
        local b_evt_len = buffer(offset + 1, 2)
        local b_evt_cnt
        local b_evt_ep
        local evt_id = b_evt_type:uint()
        local evt_len = b_evt_len:uint()
        local evt_cnt = 0
        local evt_type_cnt_label

        evt_num = evt_num + 1

        if evt_id == 0xc0 then
            b_evt_cnt = buffer(offset + 5, 8)
            b_evt_ep = buffer(offset + 13, 8)
            evt_cnt = b_evt_cnt:uint64()
            evt_type_cnt_label = "Frm Num"
        end

        if evt_id == 0xc1 then
            b_evt_cnt = buffer(offset + 3, 8)
            b_evt_ep = buffer(offset + 11, 8)
            evt_cnt = b_evt_cnt:uint64()
            evt_type_cnt_label = "Lifesign Num"
        end

        local timestamp = os.date("%b %d, %Y %X", 
                            b_evt_ep:uint64():tonumber() / 1000000) .. 
                            "." .. 
                            b_evt_ep:uint64():tonumber() % 1000000

        local t_evt = t_fwp:add(buffer(offset, evt_len), 
                            "Event " .. tostring(evt_num) .. ": " ..
                            vs_evt_type[evt_id] .. ", " ..
                            evt_type_cnt_label .. ": " ..
                            tostring(evt_cnt) .. ", Timestamp: " ..
                            timestamp)

        -- event header
        local t_evt_hdr = t_evt:add(b_evt_hdr, "Event Header:"):append_text(
                            " Type: " .. vs_evt_type[b_evt_type:uint()] ..
                            ", Length: " .. b_evt_len:uint64() .. " bytes")
        t_evt_hdr:add(f_evt_type, b_evt_type)
        t_evt_hdr:add(f_evt_len, b_evt_len):append_text(" bytes")

        -- event: lifesign, system monitoring
        if evt_id == 0xc1 then

            local b_sys_cnt = buffer(offset + 3, 8)
            local b_sys_ep = buffer(offset + 11, 8)
            local b_sys_sn = buffer(offset + 19, 4)
            local b_cpu = buffer(offset + 23, 8)
            local b_cpu_load = buffer(offset + 23, 4)
            local b_cpu_temp = buffer(offset + 27, 4)
            local b_mmc = buffer(offset + 31, 16)
            local b_mmc_total = buffer(offset + 31, 4)
            local b_mmc_used = buffer(offset + 35, 4)
            local b_mmc_avail = buffer(offset + 39, 4)
            local b_mmc_used_per = buffer(offset + 43, 4)
            local b_ram = buffer(offset + 47, 16)
            local b_ram_total = buffer(offset + 47, 4)
            local b_ram_used = buffer(offset + 51, 4)
            local b_ram_avail = buffer(offset + 55, 4)
            local b_ram_used_per = buffer(offset + 59, 4)
            local b_buf = buffer(offset + 63, 16)
            local b_buf_total = buffer(offset + 63, 4)
            local b_buf_used = buffer(offset + 67, 4)
            local b_buf_avail = buffer(offset + 71, 4)
            local b_buf_used_per = buffer(offset + 75, 4)
            local b_udp = buffer(offset + 79, 91)
            local b_udp_act = buffer(offset + 79, 51)
            local b_udp_is_on = buffer(offset + 79, 1)
            local b_udp_total_on = buffer(offset + 80, 8)
            local b_udp_total_off = buffer(offset + 88, 8)
            local b_udp_last_start_ep = buffer(offset + 96, 8)
            local b_udp_last_stop_ep = buffer(offset + 104, 8)
            local b_udp_last_on = buffer(offset + 112, 8)
            local b_udp_last_off = buffer(offset + 120, 8)
            local b_udp_loss = buffer(offset + 128, 2)
            local b_udp_rx = buffer(offset + 130, 20)
            local b_udp_rx_f_cnt = buffer(offset + 130, 8)
            local b_udp_rx_f_rate = buffer(offset + 138, 4)
            local b_udp_rx_b_rate = buffer(offset + 142, 8)
            local b_udp_tx = buffer(offset + 150, 20)
            local b_udp_tx_f_cnt = buffer(offset + 150, 8)
            local b_udp_tx_f_rate = buffer(offset + 158, 4)
            local b_udp_tx_b_rate = buffer(offset + 162, 8)
            local b_fip = buffer(offset + 170, 153)
            local b_fip_act = buffer(offset + 170, 51)
            local b_fip_is_on = buffer(offset + 170, 1)
            local b_fip_total_on = buffer(offset + 171, 8)
            local b_fip_total_off = buffer(offset + 179, 8)
            local b_fip_last_start_ep = buffer(offset + 187, 8)
            local b_fip_last_stop_ep = buffer(offset + 195, 8)
            local b_fip_last_on = buffer(offset + 203, 8)
            local b_fip_last_off =  buffer(offset + 211, 8)
            local b_fip_loss = buffer(offset + 219, 2)
            local b_fip_rx = buffer(offset + 221, 20)
            local b_fip_rx_f_cnt = buffer(offset + 221, 8)
            local b_fip_rx_f_rate = buffer(offset + 229, 4)
            local b_fip_rx_b_rate = buffer(offset + 233, 8)
            local b_fip_net_type = buffer(offset + 241, 1)
            local b_fip_net_bitrate = buffer(offset + 242, 1)
            local b_fip_err = buffer(offset + 243, 80)
            local b_fip_err_phy = buffer(offset + 243, 24)
            local b_fip_err_dl = buffer(offset + 267, 16)
            local b_fip_err_app = buffer(offset + 283, 16)
            local b_fip_err_hdl = buffer(offset + 299, 24)
            local b_fip_err_phy_pre = buffer(offset + 243, 8)
            local b_fip_err_phy_fsd = buffer(offset + 251, 8)
            local b_fip_err_phy_fed = buffer(offset + 259, 8)
            local b_fip_err_dl_ctl = buffer(offset + 267, 8)
            local b_fip_err_dl_fcs = buffer(offset + 275, 8)
            local b_fip_err_app_pdu_type = buffer(offset + 283, 8)
            local b_fip_err_app_pdu_len = buffer(offset + 291, 8)
            local b_fip_err_hdl_glc = buffer(offset + 299, 8)
            local b_fip_err_hdl_ifc = buffer(offset + 307, 8)
            local b_fip_err_hdl_fc = buffer(offset + 315, 8)

            local udp_last_start_tstp = os.date("%b %d, %Y %X", 
                    b_udp_last_start_ep:uint64():tonumber() / 1000000) .. 
                    "." .. 
                    b_udp_last_start_ep:uint64():tonumber() % 1000000

            local udp_last_stop_tstp = os.date("%b %d, %Y %X", 
                    b_udp_last_stop_ep:uint64():tonumber() / 1000000) .. 
                    "." .. 
                    b_udp_last_stop_ep:uint64():tonumber() % 1000000

            local fip_last_start_tstp = os.date("%b %d, %Y %X", 
                    b_fip_last_start_ep:uint64():tonumber() / 1000000) .. 
                    "." .. 
                    b_fip_last_start_ep:uint64():tonumber() % 1000000

            local fip_last_stop_tstp = os.date("%b %d, %Y %X", 
                    b_fip_last_stop_ep:uint64():tonumber() / 1000000) .. 
                    "." .. 
                    b_fip_last_stop_ep:uint64():tonumber() % 1000000

            t_evt:add(f_sys_cnt, b_sys_cnt)
            t_evt:add(f_sys_ep, b_sys_ep):append_text(" us")
            t_evt:add(b_evt_ep, "Capture Timestamp: " .. timestamp)
            t_evt:add(f_sys_sn, b_sys_sn)
            -- cpu info
            local t_cpu = t_evt:add(b_cpu, "CPU:"):append_text(
                            " Load: " .. 
                            string.format("%.1f", b_cpu_load:float()) .. "%" ..
                            ", Temperature: " .. 
                            string.format("%.1f", b_cpu_temp:float()) .. "°C")
            t_cpu:add(f_cpu_load, b_cpu_load):append_text(" %")
            t_cpu:add(f_cpu_temp, b_cpu_temp):append_text(" °C")
            -- mmc memory info
            local t_mmc = t_evt:add(b_mmc, "eMMC:"):append_text(
                            " Used: " .. 
                            string.format("%.1f", b_mmc_used_per:float()) .. "%" ..
                            ", Available: " .. b_mmc_avail:uint64() .. 
                            "KB")
            t_mmc:add(f_mmc_total, b_mmc_total):append_text(" KB")
            t_mmc:add(f_mmc_used, b_mmc_used):append_text(" KB")
            t_mmc:add(f_mmc_avail, b_mmc_avail):append_text(" KB")
            t_mmc:add(f_mmc_used_per, b_mmc_used_per):append_text(" %")
            -- ram memory info
            local t_ram = t_evt:add(b_ram, "RAM:"):append_text(
                            " Used: " .. 
                            string.format("%.1f", b_ram_used_per:float()) .. "%" ..
                            ", Available: " .. b_ram_avail:uint64() .. 
                            "KB")
            t_ram:add(f_ram_total, b_ram_total):append_text(" KB")
            t_ram:add(f_ram_used, b_ram_used):append_text(" KB")
            t_ram:add(f_ram_avail, b_ram_avail):append_text(" KB")
            t_ram:add(f_ram_used_per, b_ram_used_per):append_text(" %")
            -- buffer (fip/udp) info
            local t_buf = t_evt:add(b_buf, "FIP-UDP Buffer:"):append_text(
                            " Used: " .. 
                            string.format("%.1f", b_buf_used_per:float()) .. "%" ..
                            ", Available: " .. b_buf_avail:uint64() .. 
                            "KB")
            t_buf:add(f_buf_total, b_buf_total):append_text(" KB")
            t_buf:add(f_buf_used, b_buf_used):append_text(" KB")
            t_buf:add(f_buf_avail, b_buf_avail):append_text(" KB")
            t_buf:add(f_buf_used_per, b_buf_used_per):append_text(" %")
            -- udp infos 
            local t_udp = t_evt:add(b_udp, "UDP Infos: "):append_text(
                                vs_act_status[b_udp_is_on:uint()] ..
                                ", Tx: " .. b_udp_tx_f_rate:uint64() .. 
                                " fps (" .. b_udp_tx_b_rate:uint64() .. " B/s)")
            -- udp activity
            local t_udp_act = t_udp:add(b_udp_act, "Activity: "):append_text(
                                vs_act_status[b_udp_is_on:uint()])
            t_udp_act:add(f_udp_is_on, b_udp_is_on)
            t_udp_act:add(f_udp_total_on, b_udp_total_on):append_text(" us")
            t_udp_act:add(f_udp_total_off, b_udp_total_off):append_text(" us")
            t_udp_act:add(f_udp_last_start_ep, b_udp_last_start_ep):append_text(" us")
            t_udp_act:add(b_udp_last_start_ep, "Last Start Timestamp: " .. 
                                udp_last_start_tstp)
            t_udp_act:add(f_udp_last_stop_ep, b_udp_last_stop_ep):append_text(" us")
            t_udp_act:add(b_udp_last_stop_ep, "Last Stop Timestamp: " .. 
                                udp_last_stop_tstp)
            t_udp_act:add(f_udp_last_on, b_udp_last_on):append_text(" us")
            t_udp_act:add(f_udp_last_off, b_udp_last_off):append_text(" us")
            t_udp_act:add(f_udp_loss_cnt, b_udp_loss)
            -- udp rx
            local t_udp_rx = t_udp:add(b_udp_rx, "Rx: "):append_text(
                                b_udp_rx_f_rate:uint64() .. " fps (" .. 
                                b_udp_rx_b_rate:uint64() .. " B/s)")
            t_udp_rx:add(f_udp_rx_f_count, b_udp_rx_f_cnt)
            t_udp_rx:add(f_udp_rx_f_rate, b_udp_rx_f_rate):append_text(" fps")
            t_udp_rx:add(f_udp_rx_b_rate, b_udp_rx_b_rate):append_text(" B/s")
            -- udp tx
            local t_udp_tx = t_udp:add(b_udp_tx, "Tx: "):append_text(
                                b_udp_tx_f_rate:uint64() .. " fps (" .. 
                                b_udp_tx_b_rate:uint64() .. " B/s)")
            t_udp_tx:add(f_udp_tx_f_count, b_udp_tx_f_cnt)
            t_udp_tx:add(f_udp_tx_f_rate, b_udp_tx_f_rate):append_text(" fps")
            t_udp_tx:add(f_udp_tx_b_rate, b_udp_tx_b_rate):append_text(" B/s")
            -- fip infos
            local t_fip = t_evt:add(b_fip, "FIP Infos: "):append_text(
                                vs_act_status[b_fip_is_on:uint()] ..
                                ", Rx: " .. b_fip_rx_f_rate:uint64() .. 
                                " fps (" .. b_fip_rx_b_rate:uint64() .. " B/s)")
            -- fip activity
            local t_fip_act = t_fip:add(b_fip_act, "Activity: "):append_text(
                                vs_act_status[b_fip_is_on:uint()])
            t_fip_act:add(f_fip_is_on, b_fip_is_on)
            t_fip_act:add(f_fip_total_on, b_fip_total_on):append_text(" us")
            t_fip_act:add(f_fip_total_off, b_fip_total_off):append_text(" us")
            t_fip_act:add(f_fip_last_start_ep, b_fip_last_start_ep):append_text(" us")
            t_fip_act:add(b_fip_last_start_ep, "Last Start Timestamp: " .. 
                                fip_last_start_tstp)
            t_fip_act:add(f_fip_last_stop_ep, b_fip_last_stop_ep):append_text(" us")
            t_fip_act:add(b_fip_last_stop_ep, "Last Stop Timestamp: " .. 
                                fip_last_stop_tstp)
            t_fip_act:add(f_fip_last_on, b_fip_last_on):append_text(" us")
            t_fip_act:add(f_fip_last_off, b_fip_last_off):append_text(" us")
            t_fip_act:add(f_fip_loss_cnt, b_fip_loss)
            -- fip rx
            local t_fip_rx = t_fip:add(b_fip_rx, "Rx: "):append_text(
                                b_fip_rx_f_rate:uint64() .. " fps (" .. 
                                b_fip_rx_b_rate:uint64() .. " B/s)")
            t_fip_rx:add(f_fip_rx_f_count, b_fip_rx_f_cnt)
            t_fip_rx:add(f_fip_rx_f_rate, b_fip_rx_f_rate):append_text(" fps")
            t_fip_rx:add(f_fip_rx_b_rate, b_fip_rx_b_rate):append_text(" B/s")
            -- fip network type
            t_fip:add(f_fip_net_type, b_fip_net_type)
            t_fip:add(f_fip_net_bitrate, b_fip_net_bitrate)
            -- fip errors
            local total_fip_err_phy = b_fip_err_phy_pre:uint64():tonumber()
                                    + b_fip_err_phy_fsd:uint64():tonumber()
                                    + b_fip_err_phy_fed:uint64():tonumber()
            local total_fip_err_dl = b_fip_err_dl_ctl:uint64():tonumber()
                                    + b_fip_err_dl_fcs:uint64():tonumber()
            local total_fip_err_app = b_fip_err_app_pdu_type:uint64():tonumber()
                                    + b_fip_err_app_pdu_len:uint64():tonumber()
            local total_fip_err_hdl = b_fip_err_hdl_glc:uint64():tonumber()
                                    + b_fip_err_hdl_ifc:uint64():tonumber()
                                    + b_fip_err_hdl_fc:uint64():tonumber()
            local total_fip_err = total_fip_err_phy + total_fip_err_dl
                                + total_fip_err_app + total_fip_err_hdl

            local t_fip_err = t_fip:add(b_fip_err, 
                                "Error Counters:"):append_text(
                                " Total: " .. total_fip_err ..
                                ", PHY: " .. total_fip_err_phy ..
                                ", DL: " .. total_fip_err_dl ..
                                ", APP: " .. total_fip_err_app ..
                                ", HDL: " .. total_fip_err_hdl)

            local t_fip_err_phy = t_fip_err:add(b_fip_err_phy, 
                                "Physical Layer:"):append_text(
                                " Total: " .. total_fip_err_phy ..
                                ", PRE: " .. b_fip_err_phy_pre:uint64():tonumber() ..
                                ", FSD: " .. b_fip_err_phy_fsd:uint64():tonumber() ..
                                ", FED: " .. b_fip_err_phy_fed:uint64():tonumber())
            t_fip_err_phy:add(f_fip_err_phy_pre, b_fip_err_phy_pre)
            t_fip_err_phy:add(f_fip_err_phy_fsd, b_fip_err_phy_fsd)
            t_fip_err_phy:add(f_fip_err_phy_fed, b_fip_err_phy_fed)

            local t_fip_err_dl = t_fip_err:add(b_fip_err_dl, 
                                "DataLink Layer:"):append_text(
                                " Total: " .. total_fip_err_dl ..
                                ", CTL: " .. b_fip_err_dl_ctl:uint64():tonumber() ..
                                ", FCS: " .. b_fip_err_dl_fcs:uint64():tonumber())
            t_fip_err_dl:add(f_fip_err_dl_ctl, b_fip_err_dl_ctl)
            t_fip_err_dl:add(f_fip_err_dl_fcs, b_fip_err_dl_fcs)

            local t_fip_err_app = t_fip_err:add(b_fip_err_app, 
                                "Application Layer:"):append_text(
                                " Total: " .. total_fip_err_app ..
                                ", PDU_TYPE: " .. b_fip_err_app_pdu_type:uint64():tonumber() ..
                                ", PDU_LEN: " .. b_fip_err_app_pdu_len:uint64():tonumber())
            t_fip_err_app:add(f_fip_err_app_pdu_type, b_fip_err_app_pdu_type)
            t_fip_err_app:add(f_fip_err_app_pdu_len, b_fip_err_app_pdu_len)

            local t_fip_err_hdl = t_fip_err:add(b_fip_err_hdl, 
                                "HDL Coworker Overflows:"):append_text(
                                " Total: " .. total_fip_err_hdl ..
                                ", GLC: " .. b_fip_err_hdl_glc:uint64():tonumber() ..
                                ", IFC: " .. b_fip_err_hdl_ifc:uint64():tonumber() .. 
                                ", FC: " .. b_fip_err_hdl_fc:uint64():tonumber())
            t_fip_err_hdl:add(f_fip_err_hdl_glc, b_fip_err_hdl_glc)
            t_fip_err_hdl:add(f_fip_err_hdl_ifc, b_fip_err_hdl_ifc)
            t_fip_err_hdl:add(f_fip_err_hdl_fc, b_fip_err_hdl_fc)

            if nb_evt == 1 then
                pinfo.cols.info = "Lifesign: Num=" .. b_sys_cnt:uint64():tonumber() ..
                                ", Rx FIP=" .. vs_act_status[b_fip_is_on:uint()] ..
                                " (" .. b_fip_rx_f_rate:uint64() .. " fps)" ..
                                ", Tx UDP=" .. vs_act_status[b_udp_is_on:uint()] ..
                                " (" .. b_udp_tx_f_rate:uint64() .. " fps)"
            end
        end

        -- event: rx fip frame
        if evt_id == 0xc0 then
            local data_len = evt_len - 30
            local b_fip_frm_status = buffer(offset + 3, 2)
            local b_fip_frm_cnt = buffer(offset + 5, 8)
            local b_fip_frm_ep = buffer(offset + 13, 8)
            local b_fip_frm_dur = buffer(offset + 21, 3)
            local b_fip_inter_frm_dur = buffer(offset + 24, 3)
            local b_fip_frm_glitch_cnt = buffer(offset + 27, 2)
            local b_fip_frm_jitter = buffer(offset + 29, 1)
            local b_fip_frm = buffer(offset + 30, data_len)
            local b_fip_frm_ctrl = buffer(offset + 30, 1)
            local b_fip_frm_pdu
            local b_fip_frm_pdu_type
            local b_fip_frm_pdu_len
            local b_fip_frm_user_data
            local b_fip_frm_id
            local b_fip_frm_fcs = buffer(offset + 30 + data_len - 2, 2)

            local frm_ctrl = b_fip_frm_ctrl:uint()
            local info_str = ""

            local t_fip_frm_status = t_evt:add(f_fip_frm_status, b_fip_frm_status)
            t_fip_frm_status:add(f_fip_frm_status_err_pre, b_fip_frm_status)
            t_fip_frm_status:add(f_fip_frm_status_err_no_fsd, b_fip_frm_status)
            t_fip_frm_status:add(f_fip_frm_status_err_fsd, b_fip_frm_status)
            t_fip_frm_status:add(f_fip_frm_status_err_fed, b_fip_frm_status)
            t_fip_frm_status:add(f_fip_frm_status_err_fc, b_fip_frm_status)
            t_fip_frm_status:add(f_fip_frm_status_err_ifc, b_fip_frm_status)
            t_fip_frm_status:add(f_fip_frm_status_err_glc, b_fip_frm_status)
            t_fip_frm_status:add(f_fip_frm_status_ch_rx, b_fip_frm_status)
            t_fip_frm_status:add(f_fip_frm_status_net_type, b_fip_frm_status)
            t_fip_frm_status:add(f_fip_frm_status_net_bitrate, b_fip_frm_status)

            t_evt:add(f_fip_frm_cnt, b_fip_frm_cnt)
            t_evt:add(f_fip_frm_ep, b_fip_frm_ep):append_text(" us")
            t_evt:add(b_evt_ep, "Capture Timestamp: " .. timestamp)
            t_evt:add(f_fip_frm_dur, b_fip_frm_dur):append_text(
                " (" .. string.format("%.1f", b_fip_frm_dur:uint()/10) .. " us)")
            t_evt:add(f_fip_inter_frm_dur, b_fip_inter_frm_dur):append_text(
                " (" .. string.format("%.1f", b_fip_inter_frm_dur:uint()/10) .. " us)")
            t_evt:add(f_fip_frm_glitch_cnt, b_fip_frm_glitch_cnt)
            t_evt:add(f_fip_frm_jitter, b_fip_frm_jitter):append_text(
                " (" .. string.format("%.1f", b_fip_frm_jitter:uint()/2) .. " %)")
            
            local t_fip_frm = t_evt:add(f_fip_frm, b_fip_frm)
            
            local t_fip_frm_ctrl = t_fip_frm:add(f_fip_frm_ctrl, b_fip_frm_ctrl)
            t_fip_frm_ctrl:add(f_fip_frm_ctrl_dir, b_fip_frm_ctrl)
            t_fip_frm_ctrl:add(f_fip_frm_ctrl_var, b_fip_frm_ctrl)
            t_fip_frm_ctrl:add(f_fip_frm_ctrl_msg, b_fip_frm_ctrl)
            t_fip_frm_ctrl:add(f_fip_frm_ctrl_var_list, b_fip_frm_ctrl)
            t_fip_frm_ctrl:add(f_fip_frm_ctrl_ack_frm, b_fip_frm_ctrl)
            t_fip_frm_ctrl:add(f_fip_frm_ctrl_prio_or_ack, b_fip_frm_ctrl)
            t_fip_frm_ctrl:add(f_fip_frm_ctrl_end, b_fip_frm_ctrl)
            t_fip_frm_ctrl:add(f_fip_frm_ctrl_parity, b_fip_frm_ctrl)

            info_str = info_str .. vs_fip_frm_ctrl[frm_ctrl]
 
            if vs_fip_frm_ctrl[frm_ctrl] == "ID_DAT" or
                vs_fip_frm_ctrl[frm_ctrl] == "ID_MSG" or
                vs_fip_frm_ctrl[frm_ctrl] == "ID_RQ1" or
                vs_fip_frm_ctrl[frm_ctrl] == "ID_RQ2" then

                    b_fip_frm_id = buffer(offset + 31, 2)
                    t_fip_frm:add(f_fip_frm_id, b_fip_frm_id)

                    -- info column
                    info_str = info_str .. string.format("(0x%04x)", b_fip_frm_id:uint())

            elseif vs_fip_frm_ctrl[frm_ctrl] == "RP_DAT" or
                vs_fip_frm_ctrl[frm_ctrl] == "RP_DAT_MSG" or
                vs_fip_frm_ctrl[frm_ctrl] == "RP_DAT_RQ1" or
                vs_fip_frm_ctrl[frm_ctrl] == "RP_DAT_RQ2" or
                vs_fip_frm_ctrl[frm_ctrl] == "RP_DAT_RQ1_MSG" or
                vs_fip_frm_ctrl[frm_ctrl] == "RP_DAT_RQ2_MSG" then

                    b_fip_frm_pdu = buffer(offset + 31, 2)
                    b_fip_frm_pdu_type = buffer(offset + 31, 1)
                    b_fip_frm_pdu_len = buffer(offset + 32, 1)
                    b_fip_frm_user_data = buffer(offset + 33, b_fip_frm_pdu_len:uint())
                    b_fip_frm_data = buffer(offset + 31, 2 + b_fip_frm_pdu_len:uint())

                    local t_fip_frm_pdu = t_fip_frm:add(f_fip_frm_pdu,
                                             b_fip_frm_pdu):append_text(
                                    " (" .. 
                                    vs_fip_frm_pdu_type[b_fip_frm_pdu_type:uint()] .. 
                                    ", Length: " .. b_fip_frm_pdu_len:uint() .. 
                                    " bytes)")
                    t_fip_frm_pdu:add(f_fip_frm_pdu_type, b_fip_frm_pdu_type)
                    t_fip_frm_pdu:add(f_fip_frm_pdu_len, 
                                        b_fip_frm_pdu_len):append_text(" bytes")
                    t_fip_frm:add(f_fip_frm_user_data, b_fip_frm_user_data)

                    -- info column
                    info_str = info_str .. "("
                    for i=0,1+b_fip_frm_pdu_len:uint(),1 do
                        info_str = info_str .. string.format("%02x ", 
                                buffer(offset + 31 + i,1):uint())
                    end
                    info_str = info_str:sub(1,-2)
                    info_str = info_str .. ")"

            elseif vs_fip_frm_ctrl[frm_ctrl] == "RP_RQ1" or
                vs_fip_frm_ctrl[frm_ctrl] == "RP_RQ2" then

                    -- payload = datalen - CTRL - FCS
                    local nb_ids = (data_len - 3) / 2
                    local b_fip_frm_list_ids = buffer(offset + 31, nb_ids*2)
                    local t_fip_frm_list_ids =  t_fip_frm:add(f_fip_frm_list_ids, 
                                                    b_fip_frm_list_ids)
                    for i=0,nb_ids-1,1 do
                        local b_rq_id = buffer(offset + 31 + i*2, 2)
                        t_fip_frm_list_ids:add(b_rq_id, string.format('[%d]: 0x%04x', 
                            i+1, b_rq_id:uint()))
                    end

                    -- info column
                    info_str = info_str .. "("
                    for i=0,nb_ids*2,1 do
                        info_str = info_str .. string.format("%02x ", 
                                buffer(offset + 31 + i,1):uint())
                    end
                    info_str = info_str:sub(1,-2)
                    info_str = info_str .. ")"

            elseif vs_fip_frm_ctrl[frm_ctrl] == "RP_MSG_NOACK" or
                vs_fip_frm_ctrl[frm_ctrl] == "RP_MSG_ACKe" or
                vs_fip_frm_ctrl[frm_ctrl] == "RP_MSG_ACKo" then

                    -- payload = datalen - CTRL - src_add - dst_add - FCS
                    local msg_len = (data_len - 9)
                    local b_fip_frm_dst_add = buffer(offset + 31, 3)
                    local b_fip_frm_src_add = buffer(offset + 34, 3)
                    b_fip_frm_user_data = buffer(offset + 37, msg_len)

                    t_fip_frm:add(f_fip_frm_dst_add, b_fip_frm_dst_add)
                    t_fip_frm:add(f_fip_frm_src_add, b_fip_frm_src_add)
                    t_fip_frm:add(f_fip_frm_user_data, b_fip_frm_user_data)

                    -- info column
                    info_str = info_str .. "("
                    for i=0,5+msg_len,1 do
                        info_str = info_str .. string.format("%02x ", 
                                buffer(offset + 31 + i,1):uint())
                    end
                    info_str = info_str:sub(1,-2)
                    info_str = info_str .. ")"

            elseif vs_fip_frm_ctrl[frm_ctrl] == "RP_ACKpe" or
                vs_fip_frm_ctrl[frm_ctrl] == "RP_ACKpo" or
                vs_fip_frm_ctrl[frm_ctrl] == "RP_ACKme" or
                vs_fip_frm_ctrl[frm_ctrl] == "RP_ACKmo" then

                -- info column
                info_str = info_str .. "()"

            elseif vs_fip_frm_ctrl[frm_ctrl] == "RP_FIN" then

                -- info column
                info_str = info_str .. "()"
            end

            t_fip_frm:add(f_fip_frm_fcs, b_fip_frm_fcs)

            -- info column is updated only if the udp packet contains
            -- an unique event
            if nb_evt == 1 then
                pinfo.cols.info = info_str
            end
        end

        offset = offset + evt_len
    end

    while offset < buffer:len() do 
        evt_pass(buffer)
    end
end

-- load the udp port table
udp_table = DissectorTable.get("udp.port")
-- register the protocol to port
udp_table:add(UDP_PORT, fwp_proto)
