/*
 * Copyright (C) 2014-2019 EXOLIGENT (www.exoligent.com)
 * Author: Maxime Coroyer <maxime.coroyer@exoligent.com>
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include <linux/pci.h>
#include <linux/interrupt.h>
#include <linux/spinlock.h>
#include <asm/io.h>
#include <linux/list.h>
#include <linux/hrtimer.h>
#include <linux/sched.h>

#include "fw-drv.h"
#include "fw-pci.h"
#include "../include/fw-board.h"

// Device ID
// For backward compatibility
// But this is not normal. Values ​​should be:
// PLX 9052: Vendor ID: 0x10b5 (PLX) - Device ID: 0x9052
// PLX 9030: Vendor ID: 0x10b5 (PLX) - Device ID: 0x9030
#define PCI_VEN_ID_PLX_9030 0x10b5
#define PCI_DEV_ID_PLX_9030 0x5201 
#define SUBSYS_DEV_ID_FW_MPCIE_COPPER 0x5802

#define DRV_NAME "pcifw"
#define DRV_HDR "pcifw: "
#define BOARD_INFO "FIP/WorldFIP bus analyzer board"

static struct class *cl = NULL; // Global variable for the device class
int pci_major = PCI_MAJOR;
int pci_devs  = PCI_DEVS;
static u8 minors[PCI_DEVS] = {0, 0, 0, 0, 0, 0, 0, 0};
static LIST_HEAD(privates);

MODULE_DESCRIPTION("FipWatcher driver for PCI/PCIe board [minimal]");
MODULE_AUTHOR("Exoligent SARL.");
MODULE_LICENSE("GPL v2");
MODULE_VERSION("1.0");

static struct pci_device_id ids[] = 
    {{.vendor = PCI_VENDOR_ID_PLX, .device = PCI_DEV_ID_PLX_9030, 
      .subvendor = PCI_VENDOR_ID_PLX, .subdevice = SUBSYS_DEV_ID_FW_MPCIE_COPPER},
    {0,}};

static struct {const char *name;} board_info[] = { {BOARD_INFO},};

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,20)
    static irqreturn_t irq_handler(int irq, void *dev_id)
#else
    static irqreturn_t irq_handler(int irq, void *dev_id, struct pt_regs *regs)
#endif
{
    int i = 0;
    u32 intcsr_reg, reg;
    u32 limit_u32 = -1;
    struct pci_priv_device *pdev = (struct pci_priv_device *) dev_id;

    reg = read_register_u32(pdev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_INTCSR);

    /* is the IRQ signal is produced by the device ? */
    if ((reg & C_MK_REG_BIT_IT_LINT1_ACTIVE) ||
        (reg & C_MK_REG_BIT_IT_LINT2_ACTIVE)) {

        /* irq produced by this device */
        /* disable IRQ */
        intcsr_reg = read_register_u32(pdev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_INTCSR);
        intcsr_reg &= ~C_MK_REG_BIT_IT_ENABLE;
        write_register_u32(pdev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_INTCSR, intcsr_reg);

        for (i = 0 ; i < MAX_FW_CORE ; i++)
            pdev->interrupts.fw_irq_flag[i] = 0;

        if (reg & C_MK_REG_BIT_IT_LINT1_ACTIVE) {
            pdev->interrupts.fw_irq_flag[0] = 1;
            pdev->interrupts.fw_irq_counter[0] += 1;
        }

        /* raise the interrupt flag + wake up the mutex */
        pdev->int_flag = 1;
        wake_up(&pdev->int_mutex);

        (pdev->irq_handled_counter == limit_u32) ? 0 : pdev->irq_handled_counter++;

        return IRQ_HANDLED;
    } else {
        /* irq not produced by this device */
        (pdev->irq_none_handled_counter == limit_u32) ? 0 : pdev->irq_none_handled_counter++;
        return IRQ_HANDLED;
    }
}

static void pci_setup_cdev(struct pci_priv_device* dev, int index)
{
    char dev_name[64];
    int err, devno = MKDEV(pci_major, index);

    /* add class to device - ls /dev/ */
    sprintf(dev_name, "%s%d", DRV_NAME, index);
    dev->dev = device_create(cl, NULL, devno, NULL, dev_name);
    if (IS_ERR(dev->dev))
        printk(KERN_ERR DRV_HDR "Error when adding class to device (index %d)", index);

    /* add char device */
    cdev_init(&dev->cdev, &pcifw_fops);
    dev->cdev.owner = THIS_MODULE;
    dev->cdev.ops = &pcifw_fops;
    err = cdev_add (&dev->cdev, devno, pci_devs);
    if (err)
        printk(KERN_ERR DRV_HDR "Error %d when adding char device (index %d)", err, index);
}

static int probe(struct pci_dev *dev, const struct pci_device_id *id)
{
    int rc = 0;
    struct pci_priv_device *priv_dev;
    int  i;
    void *mmio[DEVICE_COUNT_RESOURCE];   /* Remaped I/O memory */
    u32 mmio_len[DEVICE_COUNT_RESOURCE]; /* Size of remaped I/O memory */
    u8 irq_pin;
    u8 ioRessourceMem;
    u8 free_minor_dectected = 0;
    unsigned long flag;

    spinlock_t to_lock = __SPIN_LOCK_UNLOCKED(to_lock);
    int err = pci_enable_device(dev);
    if (err)
        return err;

    printk(KERN_DEBUG DRV_HDR "==> Device probing: %s\n", 
        board_info[id->driver_data].name);
    
    /* Allocate a private structure and reference it as driver's data */
    priv_dev = (struct pci_priv_device *) kcalloc(1, 
                        sizeof(struct pci_priv_device), GFP_KERNEL);
    if (priv_dev == NULL)
        return -ENODEV;

    priv_dev->pci_dev = dev;

    spin_lock_init(&to_lock);
    spin_lock(&to_lock);

    for (i = 0; i < PCI_DEVS; i++) {
        if (minors[i] == 0) {
            minors[i] = 1 ;
            priv_dev->type = i;
            free_minor_dectected = 1;
            break;
        }
    }

    if (free_minor_dectected == 0)
        return -EFAULT;

    spin_unlock(&to_lock);
    
    priv_dev->minor = priv_dev->type ;
    pci_set_drvdata(dev, priv_dev);
    pci_setup_cdev(priv_dev, priv_dev->minor);

    /* Reserve PCI I/O and memory resources */
    rc = pci_request_regions(dev, DRV_NAME);
    if (rc)
        return -ENODEV;

    ioRessourceMem = 0;
    for (i = 0; i < DEVICE_COUNT_RESOURCE; i++)  {

        if (pci_resource_start(dev, i) != 0) {
            printk(KERN_DEBUG DRV_HDR 
                "  BAR %d (%#08x - %#08x), len = %d, flags = %#08x\n", i,
                (u32) pci_resource_start(dev, i),
                (u32) pci_resource_end(dev, i),
                (u32) pci_resource_len(dev, i),
                (u32) pci_resource_flags(dev, i));
        }
        
        flag = pci_resource_flags(dev, i);

        if ((flag & IORESOURCE_MEM) == IORESOURCE_MEM) {

            mmio[i] = ioremap(pci_resource_start(dev, i), pci_resource_len(dev, i));
            if (mmio[i] == NULL) {
                rc = -ENOMEM;
                return rc;
            }
    
            mmio_len[i] = pci_resource_len(dev, i);

            if (ioRessourceMem == 0) {
                priv_dev->IoSpacePciPlx.bSpaceMapped        = true;
                priv_dev->IoSpacePciPlx.nSpaceRange         = mmio_len[i];
                priv_dev->IoSpacePciPlx.pSpaceBase          = mmio[i];
                priv_dev->IoSpacePciPlx.physicalAddress     = pci_resource_start(dev, i);
            } else if (ioRessourceMem == 1) { 
                priv_dev->IoSpacePciFipWatcher.bSpaceMapped   = true;
                priv_dev->IoSpacePciFipWatcher.nSpaceRange    = mmio_len[i];
                priv_dev->IoSpacePciFipWatcher.pSpaceBase     = mmio[i];
                priv_dev->IoSpacePciFipWatcher.physicalAddress= pci_resource_start(dev, i);
            }
            ioRessourceMem++;
        } else {
            mmio[i] = NULL;
            if ((flag & IORESOURCE_IO) == IORESOURCE_IO) {
                priv_dev->IoPortPciPlxAddress = pci_resource_start(dev, i);
                priv_dev->IoPortPciPlxSize = pci_resource_len(dev, i);
            }
        }
    }

    /* Lock init */
    spin_lock_init(&priv_dev->lock);

    rc = pci_read_config_byte(dev, PCI_INTERRUPT_PIN, &irq_pin);
    if (rc)
        return rc;

    printk(KERN_DEBUG DRV_HDR "  IRQ pin  = #%d (0=none, 1=INTA#...4=INTD#)\n", irq_pin);
    printk(KERN_DEBUG DRV_HDR "  IRQ line = #%d\n", dev->irq);

    /* Init INTCSR register */
    init_interrupt(priv_dev);

    /* Register IRQ */
    priv_dev->int_flag = 0;
    init_waitqueue_head (&priv_dev->int_mutex);

    rc = request_irq(dev->irq, &irq_handler, IRQF_SHARED, DRV_NAME, priv_dev);
    if (rc)
        return rc;

    /* Init DEV infos - user-friendly presentation */
    rc = init_dev_infos(priv_dev);
    if (rc)
        return rc;

    /* Reset IRQ counter's */
    priv_dev->irq_handled_counter = 0;
    priv_dev->irq_none_handled_counter = 0;

    spin_lock(&to_lock);
    list_add_tail(&priv_dev->link, &privates);
    spin_unlock(&to_lock);

    return 0;
}

static void remove(struct pci_dev *dev)
{
    spinlock_t to_lock = __SPIN_LOCK_UNLOCKED(to_lock);
    u32 reg;
    
    struct pci_priv_device *priv_dev = pci_get_drvdata(dev);

    printk(KERN_DEBUG DRV_HDR "==> Device removing (idx=%d): %s\n", 
           priv_dev->minor, board_info[0].name);

    /* free IRQ */
    if (priv_dev->infos.irq_number >= 0) {
        printk(KERN_DEBUG DRV_HDR "  Freeing IRQ #%d\n", priv_dev->infos.irq_number);

        reg = read_register_u32(priv_dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_INTCSR);
        reg &= ~C_MK_REG_BIT_IT_ENABLE;
        write_register_u32(priv_dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_INTCSR, reg);

        free_irq(priv_dev->infos.irq_number, (void*) priv_dev);
    }

    if (priv_dev->IoSpacePciPlx.bSpaceMapped)
        iounmap(priv_dev->IoSpacePciPlx.pSpaceBase);

    if (priv_dev->IoSpacePciFipWatcher.bSpaceMapped)
        iounmap(priv_dev->IoSpacePciFipWatcher.pSpaceBase);

    spin_lock_init(&to_lock);

    spin_lock(&to_lock);
    minors[priv_dev->minor] = 0;
    spin_unlock(&to_lock);

    pci_disable_device(dev);
    pci_release_regions(dev);

    /* device destroying */
    if (cl) {
        int devno = MKDEV(pci_major, priv_dev->minor);
        device_destroy(cl, devno);
    }

    cdev_del(&priv_dev->cdev);

    spin_lock(&to_lock);
    list_del(&priv_dev->link);
    spin_unlock(&to_lock);

    kfree(priv_dev);
}

static struct pci_driver pci_drv = {
    .name     = DRV_NAME,
    .id_table = ids,
    .probe    = probe,
    .remove   = remove
};

MODULE_DEVICE_TABLE (pci, ids);

static int __init pcifw_init(void)
{
    int err;
    dev_t dev;

    printk(KERN_DEBUG DRV_HDR "==> Init: %s\n", board_info[0].name);

    /* create device class - ls /sys/class */
    cl = class_create(THIS_MODULE, DRV_NAME);

    /* allocate device numbers - cat /proc/devices */
    err = alloc_chrdev_region(&dev, 0, pci_devs, DRV_NAME);
    if (!err) {
        pci_major = MAJOR(dev);

        /* registering driver */
        err = pci_register_driver(&pci_drv);
    }
    return err;
}

static void __exit pcifw_exit(void)
{
    printk(KERN_DEBUG DRV_HDR "==> Exit: %s\n", board_info[0].name);

    pci_unregister_driver(&pci_drv);

    if (cl)
        class_destroy(cl);

    unregister_chrdev_region(MKDEV(pci_major, 0), pci_devs);
}

ssize_t pcifw_read(struct file *file, char *buf, size_t count, loff_t *ppos)
{
    int real;
    u8* ptr;
    int i;
    struct pci_priv_device* priv_dev = file->private_data;

    if (priv_dev == 0)
        return -1;

    /* Check for overflow */
    if (count <= priv_dev->IoSpacePciPlx.nSpaceRange - (int) *ppos)
        real = count;
    else
        real = priv_dev->IoSpacePciPlx.nSpaceRange - (int) *ppos;
    
    ptr = (u8*) priv_dev->IoSpacePciPlx.pSpaceBase;
    
    if (real) {
        for (i = 0; i < real; i++) {
            int error;
            u8 c = (u8) ioread8((u8*) ptr);
            ptr++;
            error = put_user(c, buf);
            if (error)
                break;
            (char*) buf++;
        }   
        *ppos += real;
    }

    return real;
}

ssize_t pcifw_write(struct file *file, const char *buf, size_t count, loff_t *ppos)
{
    int real;
    struct pci_priv_device *priv_dev = file->private_data;

    if (priv_dev == 0)
        return 0;

    /* Check for overflow */
    if (count <= priv_dev->IoSpacePciPlx.nSpaceRange - (int) *ppos)
        real = count;
    else
        real = priv_dev->IoSpacePciPlx.nSpaceRange - (int) *ppos;

    if (real)
        copy_from_user((char*)priv_dev->IoSpacePciPlx.pSpaceBase + (int) *ppos, 
            buf, real);

    *ppos += real;

    return real;
}

int pcifw_open(struct inode *inode, struct file *file)
{
    int minor = MINOR(inode->i_rdev);
    struct list_head *cur;
    struct pci_priv_device *priv_dev;

    list_for_each(cur, &privates) {

        priv_dev = list_entry(cur, struct pci_priv_device, link);
        if (priv_dev->minor == minor) {

            if (priv_dev->is_open) {
                printk(KERN_WARNING DRV_HDR "Device is already open (index %d)\n", 
                    priv_dev->minor);
                return -EBUSY;
            }

            init_interrupt(priv_dev);
            file->private_data = priv_dev;

            priv_dev->is_open = 1;

            return 0;
        }
    }

    return -ENODEV;
}

int pcifw_release(struct inode *inode, struct file *file)
{
    struct pci_priv_device *priv_dev;

    if (file == NULL)
        return 0;

    if (file->private_data == NULL)
        return 0;

    priv_dev = file->private_data;
    priv_dev->is_open = 0;
    file->private_data = 0;

    return 0;
}

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,35))
    int pcifw_ioctl(struct inode *inode, struct file *file, 
        unsigned int cmd, unsigned long arg)
#else
    long pcifw_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
#endif
{
    struct pci_priv_device *priv_dev;
    struct arg_io args;
    struct arg_io __user *args_ptr = 0;
    void*  input_ptr = NULL;
    void*  output_ptr = NULL;
    u32    input_size = 0, output_size = 0;
    u32    status, err, io_status = 0;
    
    status     = 0;
    io_status  = 0;

    if (arg != 0) {

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(5, 0, 0))
        if (!access_ok(arg, sizeof(u32*)))
#else
        if (!access_ok(VERIFY_WRITE, arg, sizeof(u32*)))
#endif
            return -EFAULT;

        if (copy_from_user(&args, (void*) arg, sizeof(args)))
            return -EFAULT;

        args_ptr = (struct arg_io __user *) arg;
        input_size  = args.lgIn;
        output_size = args.lgOut;

        if (input_size) {
            input_ptr = kmalloc(input_size * sizeof(u8), GFP_KERNEL);
            err = copy_from_user(input_ptr, args.argIO_In, input_size);
            if (err)
                return -EFAULT;
        } 

        output_ptr = args.argIO_Out;
    }

    priv_dev = file->private_data;
    
    switch (cmd) {

    // IOCTL EDA (ExoDevice Access)
    // --- Management
    case IOCTL_EDA_GET_DEV_INFO: // O
    {
        str_exo_fwdev tmp;

        if (output_size < sizeof(str_exo_fwdev)) {
            status = -EINVAL;
            break;
        }

        memset(&tmp, 0, sizeof(tmp));
        memcpy((uint8_t*) &tmp, (uint8_t*) &(priv_dev->infos), 
            sizeof(str_exo_fwdev));

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(5, 0, 0))
        if (!access_ok(output_ptr, sizeof(str_exo_fwdev)))
#else
        if (!access_ok(VERIFY_WRITE, output_ptr, sizeof(str_exo_fwdev)))
#endif
            return -EFAULT;

        if (copy_to_user((u8*) output_ptr, (u8*) &tmp, sizeof(tmp)))
            return -EFAULT;

        io_status = sizeof(str_exo_fwdev);
        status    = 0;

    } break;

    // --- FullFip access
    case IOCTL_EDA_FW_RESET: // none
    {
        fipwatcher_reset(priv_dev, 0);
        status = 0;
    } break;

    // --- IRQ management
    case IOCTL_EDA_IRQ_WAIT_FOR: // O
    {
        u32 reg = 0;
        str_irq tmp;
        memset(&tmp, 0, sizeof(tmp));

        reg = read_register_u32(priv_dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_INTCSR);
        /* irq enable */
        reg |= C_MK_REG_BIT_IT_ENABLE;
        write_register_u32(priv_dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_INTCSR, reg);

        wait_event(priv_dev->int_mutex, priv_dev->int_flag);

        if (!(priv_dev->int_flag)) { 
            status = -EINTR;
            printk(KERN_ERR DRV_HDR "IOCTL_EDA_IRQ_WAIT_FOR - int_flag\n");
            break;
        }
        /* drop the interrupt flag + wake up the mutex */
        priv_dev->int_flag = 0;
        wake_up(&priv_dev->int_mutex);

        // ... fill flags
        tmp.fw_irq_flag = priv_dev->interrupts.fw_irq_flag[0];
        // ... fill counters
        tmp.fw_irq_counter = priv_dev->interrupts.fw_irq_counter[0];

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(5, 0, 0))
        if (!access_ok(output_ptr, sizeof(tmp)))
#else
        if (!access_ok(VERIFY_WRITE, output_ptr, sizeof(tmp)))
#endif
            return -EFAULT;

        if (copy_to_user((u8*) output_ptr, (u8*) &tmp, sizeof(tmp)))
            return -EFAULT;

        io_status = sizeof(tmp);
        status    = 0;
    } break;

    case IOCTL_EDA_IRQ_RESUME: // none
    {
        /* wake up 'manually' the mutex */
        /* useful to unlock a wait on an event that won't come */
        u32 reg = read_register_u32(priv_dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_INTCSR);
        /* irq disable */
        reg &= ~C_MK_REG_BIT_IT_ENABLE;
        write_register_u32(priv_dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_INTCSR, reg);

        /* raise the interrupt flag + wake up the mutex */
        priv_dev->int_flag = 1;
        wake_up(&priv_dev->int_mutex);
        
        status = 0;
    } break;


    default:
        printk(KERN_ERR DRV_HDR "Unsupported IOCTL command (0x%x)\n", cmd);
        status = -EINVAL;
    }

    if (input_ptr != NULL)
        kfree(input_ptr);

    if (args_ptr != NULL) {
        if (put_user(io_status, (u32*) (&args_ptr->IoStatus)))
            return -EFAULT;
    }

    return status;
}

struct file_operations pcifw_fops = {
    .owner = THIS_MODULE,
    .read  = pcifw_read,
    .write = pcifw_write,
    .open  = pcifw_open,
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,35))
    .ioctl = pcifw_ioctl,
#else
    .unlocked_ioctl = pcifw_ioctl,
#endif
    .release = pcifw_release
};

module_init(pcifw_init);
module_exit(pcifw_exit);