/*
 * Copyright (C) 2014-2019 EXOLIGENT (www.exoligent.com)
 * Author: Maxime Coroyer <maxime.coroyer@exoligent.com>
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/param.h>
#include <linux/jiffies.h>
#include <linux/uaccess.h>
#include "fw-pci.h"
#include "../include/fw-board.h"

#define DELAY 1

u8 read_register_u8(void* base_address, u8 offset)
{
    u8 c;
    u8* ptr = (u8*) base_address;
    ptr += offset;
    c = (u8) ioread8((void*) ptr);
    return c;
}

u32 read_register_u32(void* base_address, u8 offset)
{
    u32 c;
    u8* ptr = (u8*) base_address;
    ptr += offset;
    c = (u32) ioread32((void*) ptr);
    return c;
}

void write_register_u8(void* base_address, u8 offset, u8 val)
{
    u8* ptr = (u8*) base_address;
    ptr += offset;
    iowrite8(val, (void*) ptr);
}

void write_register_u32(void* base_address, u8 offset, u32 val)
{
    u8* ptr = (u8*) base_address;
    ptr += offset;
    iowrite32(val, (void*) ptr);
}

void wait_for_us(u32 delay_us)
{
    unsigned long delay_jiffies;
    bool wait;

    delay_jiffies = jiffies + usecs_to_jiffies(delay_us);
    wait = true;
    do {
        if (time_after(jiffies, delay_jiffies))
            wait = false;
    } while (wait);

}

u8 wait_for(void* addr, u8 reg, u8 status, u32 timeout_us, u8 trigger_edge)
{
    unsigned long timeoutJiffies;
    u8 status_to_check;
    u8 status_read;

    timeoutJiffies = jiffies + usecs_to_jiffies(timeout_us);

    if (trigger_edge == 1)
        status_to_check = status;
    else
        status_to_check = 0x00;

    status_read = read_register_u8(addr, reg);

    while ((status_read & status) != status_to_check) {

        if (time_after(jiffies, timeoutJiffies))
            return -ETIME;

        status_read = read_register_u8(addr, reg);
    }

    return 0;
}

void eeprom_enable(struct pci_priv_device *dev)
{
    u32 data;
    data = read_register_u32(dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_CNTRL);
    data |= C_MK_REG_BIT_EEPROM_CS_HIGH;
    write_register_u32(dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_CNTRL, data);
    data = read_register_u32(dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_CNTRL);
}

void eeprom_disable(struct pci_priv_device *dev)
{
    u32 data;
    data = read_register_u32(dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_CNTRL);
    data &= ~C_MK_REG_BIT_EEPROM_CS_HIGH;
    write_register_u32(dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_CNTRL, data);
}

void eeprom_do_clock_cycle(struct pci_priv_device *dev)
{
    u32 data;
    data = read_register_u32(dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_CNTRL);
    data &= ~C_MK_REG_BIT_EEPROM_CLOCK_HIGH;
    write_register_u32(dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_CNTRL, data);
    data = read_register_u32(dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_CNTRL);
    data |= C_MK_REG_BIT_EEPROM_CLOCK_HIGH;
    write_register_u32(dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_CNTRL, data);
    data = read_register_u32(dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_CNTRL);
    data &= ~C_MK_REG_BIT_EEPROM_CLOCK_HIGH;
    write_register_u32(dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_CNTRL,data);
}

u8 eeprom_read(struct pci_priv_device *dev, u8 *buffer, u8 start_address, u8 length)
{
    // start_address : address of the 1st 16-bit word to read
    // length : amount of 16-bit words to read
    u32 data;
    u8 mask, counter, i, j, k;
    u8 jo;

    counter = 0;
    eeprom_enable(dev);

    eeprom_write_data_bit_high(dev); /* Q = 1 */
    eeprom_write_data_bit_high(dev); /* Q = 1 */
    eeprom_write_data_bit_low(dev);  /* Q = 0 */

    mask = 0x80; /* mask = byte to send (A7 in first) */
    for (k = 0; k < 8; k++) {
        data = start_address & mask;
        (data == mask) ? eeprom_write_data_bit_high(dev) : /* Q = 1 */
                         eeprom_write_data_bit_low(dev);   /* Q = 0 */
        mask /= 2;
    }

    /* dummy bit test (should be 0) */
    jo = 0;
    while ((eeprom_read_data_bit_no_clock(dev) != 0) && (jo < 10))
        jo++;

    /* reading loop */
    for (i = 0; i < (2 * length); i++) {
        mask = 0x80;
        for (j = 8; j >= 1; j--) {
            data = eeprom_read_data_bit(dev);
            if (data == 0x01) /* bit read is 1 */
                *(buffer + i) |= mask;
            mask /= 2;
        }
        counter++;
    }
    eeprom_disable(dev);

    return counter;
}

void eeprom_write_data_bit_high(struct pci_priv_device *dev)
{
    u32 data;
    data = read_register_u32(dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_CNTRL);
    data |= C_MK_REG_BIT_EEPROM_WRITE_HIGH;
    write_register_u32(dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_CNTRL, data);
    eeprom_do_clock_cycle(dev);
}

void eeprom_write_data_bit_low(struct pci_priv_device *dev)
{
    u32 data;
    data = read_register_u32(dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_CNTRL);
    data &= ~C_MK_REG_BIT_EEPROM_WRITE_HIGH;
    write_register_u32(dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_CNTRL, data);
    eeprom_do_clock_cycle(dev);
}

u8 eeprom_read_data_bit(struct pci_priv_device *dev)
{
    u32 data;
    u8 result;
    eeprom_do_clock_cycle(dev);
    data = read_register_u32(dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_CNTRL);
    result = (data & C_MK_REG_BIT_EEPROM_READ_BIT) ? 0x01 : 0x00;
    return result;
}

u8 eeprom_read_data_bit_no_clock(struct pci_priv_device *dev)
{
    u32 data;
    u8 result;
    data = read_register_u32(dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_CNTRL);
    result = (data & C_MK_REG_BIT_EEPROM_READ_BIT) ? 0x01 : 0x00;
    return result;
}

void init_interrupt(struct pci_priv_device *dev)
{
    u32 reg;

    reg = read_register_u32(dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_INTCSR);

    /* enable LINTi1 (IRQn) and disable LINTi2 */
    reg |= C_MK_REG_BIT_IT_LINT1_ENABLE;
    reg &= ~C_MK_REG_BIT_IT_LINT2_ENABLE;

    /* LINTi1 (IRQn) - polarity 0: active low */
    reg &= ~C_MK_REG_BIT_IT_LINT1_POLARITY_1;

    /* LINTi1 (IRQn) - level triggerable interrupt */
    reg &= ~C_MK_REG_BIT_IT_LINT1_EDGE_ENABLE;

    /* local edge triggerable interrupt clear*/
    reg |= C_MK_REG_BIT_IT_LINT1_EDGE_CLEAR;

    /* PCI interrupt enable */
    reg |= C_MK_REG_BIT_IT_ENABLE;

    write_register_u32(dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_INTCSR, reg);
}

int init_dev_infos(struct pci_priv_device *dev)
{
    u32 reg;
    u32 sn;

    // How many resources (fipwatcher fifo) for this device ?
    reg = read_register_u32(dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_RESOURCES);

    dev->infos.nb_fw_mem = (int) (reg & C_MK_REG_RES_FIPWATCHER_NB);

    dev->infos.plx_base = (unsigned long) dev->IoSpacePciPlx.physicalAddress;
    dev->infos.plx_size = dev->IoSpacePciPlx.nSpaceRange;
    dev->infos.fw_mem_base = (unsigned long) dev->IoSpacePciFipWatcher.physicalAddress;
    dev->infos.fw_mem_size = dev->IoSpacePciFipWatcher.nSpaceRange;

    dev->infos.vid = dev->pci_dev->vendor;
    dev->infos.did = dev->pci_dev->device;
    dev->infos.ssvid = dev->pci_dev->subsystem_vendor;
    dev->infos.ssid = dev->pci_dev->subsystem_device;
    dev->infos.irq_number = (int) dev->pci_dev->irq;

    // Get product number - read eeprom at 0x5e address (4 bytes)
    eeprom_read(dev, (u8*) &sn, 0x5e, sizeof(sn)/2);
    dev->infos.sn = sn;

    return 0;
}

void fipwatcher_reset(struct pci_priv_device *dev, int fw_num)
{
    u32 reg;

    // pci reset
    reg = read_register_u32(dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_CNTRL);
    reg |= C_MK_REG_BIT_LOCAL_RESET_ON;
    write_register_u32(dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_CNTRL, reg);
    wait_for_us(1000);   // 1ms waiting
    reg &= ~C_MK_REG_BIT_LOCAL_RESET_ON;
    write_register_u32(dev->IoSpacePciPlx.pSpaceBase, C_REG_PLX_CNTRL, reg);
    wait_for_us(100000); // 100ms waiting

    /* reset interruptions counters relative to the specified fipwatcher */
    dev->interrupts.fw_irq_flag[fw_num] = 0;
    dev->interrupts.fw_irq_counter[fw_num] = 0;
    dev->int_flag = 0;
}